"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkForPlatformFailure = checkForPlatformFailure;
exports.handleCommitError = handleCommitError;
exports.bulkChangesDisallowed = bulkChangesDisallowed;
const error_messages_1 = require("../../constants/error-messages");
const logger_1 = require("../../logger");
const external_host_error_1 = require("../../types/errors/external-host-error");
const env_1 = require("../env");
function checkForPlatformFailure(err) {
    if ((0, env_1.getEnv)().NODE_ENV === 'test') {
        return null;
    }
    const externalHostFailureStrings = [
        'remote: Invalid username or password',
        'gnutls_handshake() failed',
        'The requested URL returned error: 403',
        'The requested URL returned error: 5',
        'The remote end hung up unexpectedly',
        'access denied or repository not exported',
        'Could not write new index file',
        'Failed to connect to',
        'Connection timed out',
        'malformed object name',
        'Could not resolve host',
        'early EOF',
        'fatal: bad config', // .gitmodules problem
        'expected flush after ref listing',
    ];
    for (const errorStr of externalHostFailureStrings) {
        if (err.message.includes(errorStr)) {
            logger_1.logger.debug({ err }, 'Converting git error to ExternalHostError');
            return new external_host_error_1.ExternalHostError(err, 'git');
        }
        /* v8 ignore start -- TODO: add test */
    }
    const configErrorStrings = [
        {
            error: 'GitLab: Branch name does not follow the pattern',
            message: "Cannot push because branch name does not follow project's push rules",
        },
        {
            error: 'GitLab: Commit message does not follow the pattern',
            message: "Cannot push because commit message does not follow project's push rules",
        },
        {
            error: ' is not a member of team',
            message: 'The `Restrict commits to existing GitLab users` rule is blocking Renovate push. Check the Renovate `gitAuthor` setting',
        },
        {
            error: 'TF401027:',
            message: 'You need the Git `GenericContribute` permission to perform this action',
        },
        {
            error: 'matches more than one',
            message: "Renovate cannot push branches if there are tags with names the same as Renovate's branches. Please remove conflicting tag names or change Renovate's branchPrefix to avoid conflicts.",
        },
    ];
    for (const { error, message } of configErrorStrings) {
        if (err.message.includes(error)) {
            logger_1.logger.debug({ err }, 'Converting git error to CONFIG_VALIDATION error');
            const res = new Error(error_messages_1.CONFIG_VALIDATION);
            res.validationError = message;
            res.validationMessage = `\`${err.message.replaceAll('`', "'")}\``;
            return res;
        }
    }
    return null;
}
/* v8 ignore stop */
/* v8 ignore start -- TODO: add tests */
function handleCommitError(err, branchName, files) {
    checkForPlatformFailure(err);
    if (err.message.includes(`'refs/heads/renovate' exists`)) {
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = 'None';
        error.validationError = 'An existing branch is blocking Renovate';
        error.validationMessage = `Renovate needs to create the branch \`${branchName}\` but is blocked from doing so because of an existing branch called \`renovate\`. Please remove it so that Renovate can proceed.`;
        throw error;
    }
    if (err.message.includes('refusing to allow a GitHub App to create or update workflow')) {
        logger_1.logger.warn('App has not been granted permissions to update Workflows - aborting branch.');
        return null;
    }
    if ((err.message.includes('remote rejected') || err.message.includes('403')) &&
        files?.some((file) => file.path?.startsWith('.github/workflows/'))) {
        logger_1.logger.debug({ err }, 'commitFiles error');
        logger_1.logger.info('Workflows update rejection - aborting branch.');
        return null;
    }
    if (err.message.includes('protected branch hook declined')) {
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = branchName;
        error.validationError = 'Renovate branch is protected';
        error.validationMessage = `Renovate cannot push to its branch because branch protection has been enabled.`;
        throw error;
    }
    if (err.message.includes('can only push your own commits')) {
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = branchName;
        error.validationError = 'Bitbucket committer error';
        error.validationMessage = `Renovate has experienced the following error when attempting to push its branch to the server: \`${err.message.replaceAll('`', "'")}\``;
        throw error;
    }
    if (err.message.includes('remote: error: cannot lock ref')) {
        logger_1.logger.error({ err }, 'Error committing files.');
        return null;
    }
    if (err.message.includes('denying non-fast-forward') ||
        err.message.includes('GH003: Sorry, force-pushing')) {
        logger_1.logger.debug({ err }, 'Permission denied to update branch');
        const error = new Error(error_messages_1.CONFIG_VALIDATION);
        error.validationSource = branchName;
        error.validationError = 'Force push denied';
        error.validationMessage = `Renovate is unable to update branch(es) due to force pushes being disallowed.`;
        throw error;
    }
    logger_1.logger.debug({ err }, 'Unknown error committing files');
    // We don't know why this happened, so this will cause bubble up to a branch error
    throw err;
}
/* v8 ignore stop */
function bulkChangesDisallowed(err) {
    return err.message.includes('update more than');
}
//# sourceMappingURL=error.js.map