"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setEmojiConfig = setEmojiConfig;
exports.emojify = emojify;
exports.stripHexCode = stripHexCode;
exports.unemojify = unemojify;
exports.stripEmojis = stripEmojis;
const tslib_1 = require("tslib");
const emoji_regex_1 = tslib_1.__importDefault(require("emoji-regex"));
const emojibase_1 = require("emojibase");
const emoji_js_1 = tslib_1.__importDefault(require("emojibase-regex/emoji.js"));
const shortcode_js_1 = tslib_1.__importDefault(require("emojibase-regex/shortcode.js"));
const zod_1 = require("zod");
const data_files_generated_1 = tslib_1.__importDefault(require("../data-files.generated"));
const logger_1 = require("../logger");
const regex_1 = require("./regex");
const result_1 = require("./result");
const schema_utils_1 = require("./schema-utils");
let unicodeEmoji = true;
let mappingsInitialized = false;
const shortCodesByHex = new Map();
const hexCodesByShort = new Map();
const EmojiShortcodesSchema = schema_utils_1.Json.pipe(zod_1.z.record(zod_1.z.string(), zod_1.z.union([zod_1.z.string().transform((val) => [val]), zod_1.z.array(zod_1.z.string())])));
const patchedEmojis = {
    '26A0-FE0F': ['warning'], // Colorful warning (⚠️) instead of black and white (⚠)
};
function initMapping(mapping) {
    for (const [hex, shortcodes] of Object.entries(mapping)) {
        const mainShortcode = `:${shortcodes[0]}:`;
        shortCodesByHex.set(hex, mainShortcode);
        shortCodesByHex.set(stripHexCode(hex), mainShortcode);
        for (const shortcode of shortcodes) {
            hexCodesByShort.set(`:${shortcode}:`, hex);
        }
    }
}
function lazyInitMappings() {
    if (!mappingsInitialized) {
        const githubShortcodes = data_files_generated_1.default.get('node_modules/emojibase-data/en/shortcodes/github.json');
        result_1.Result.parse(githubShortcodes, EmojiShortcodesSchema)
            .onValue((data) => {
            initMapping(data);
            initMapping(patchedEmojis);
            mappingsInitialized = true;
        })
            .onError(
        /* istanbul ignore next */
        (error) => {
            logger_1.logger.warn({ error }, 'Unable to parse emoji shortcodes');
        });
    }
}
function setEmojiConfig(config) {
    unicodeEmoji = !!config.unicodeEmoji;
}
const shortCodeRegex = (0, regex_1.regEx)(shortcode_js_1.default.source, 'g');
function emojify(text) {
    if (!unicodeEmoji) {
        return text;
    }
    lazyInitMappings();
    return text.replace(shortCodeRegex, (shortCode) => {
        const hexCode = hexCodesByShort.get(shortCode);
        return hexCode
            ? (0, emojibase_1.fromCodepointToUnicode)((0, emojibase_1.fromHexcodeToCodepoint)(hexCode))
            : shortCode;
    });
}
const emojiRegexSrc = [emoji_js_1.default, (0, emoji_regex_1.default)()].map(({ source }) => source);
const emojiRegex = new RegExp(`(?:${emojiRegexSrc.join('|')})`, 'g'); // TODO #12875 cannot figure it out
const excludedModifiers = new Set([
    '20E3',
    '200D',
    'FE0E',
    'FE0F',
    '1F3FB',
    '1F3FC',
    '1F3FD',
    '1F3FE',
    '1F3FF',
    '1F9B0',
    '1F9B1',
    '1F9B2',
    '1F9B3',
]);
function stripHexCode(input) {
    return input
        .split('-')
        .filter((modifier) => !excludedModifiers.has(modifier))
        .join('-');
}
function unemojify(text) {
    if (unicodeEmoji) {
        return text;
    }
    lazyInitMappings();
    return text.replace(emojiRegex, (emoji) => {
        const hexCode = stripHexCode((0, emojibase_1.fromUnicodeToHexcode)(emoji));
        const shortCode = shortCodesByHex.get(hexCode);
        return shortCode ?? /* istanbul ignore next */ '�';
    });
}
function stripEmoji(emoji) {
    const hexCode = stripHexCode((0, emojibase_1.fromUnicodeToHexcode)(emoji));
    // `hexCode` could be empty if `emoji` is a modifier character that isn't
    // modifying anything
    if (hexCode) {
        const codePoint = (0, emojibase_1.fromHexcodeToCodepoint)(hexCode);
        return (0, emojibase_1.fromCodepointToUnicode)(codePoint);
    }
    return '';
}
function stripEmojis(input) {
    return input.replace(emojiRegex, stripEmoji);
}
//# sourceMappingURL=emoji.js.map