"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.get = get;
exports.set = set;
exports.init = init;
exports.cleanup = cleanup;
const tslib_1 = require("tslib");
const env_1 = require("../../env");
const stats_1 = require("../../stats");
const memCache = tslib_1.__importStar(require("../memory"));
const fileCache = tslib_1.__importStar(require("./file"));
const key_1 = require("./key");
const redisCache = tslib_1.__importStar(require("./redis"));
const sqlite_1 = require("./sqlite");
let cacheProxy;
async function get(namespace, key) {
    if (!cacheProxy) {
        return undefined;
    }
    const combinedKey = (0, key_1.getCombinedKey)(namespace, key);
    let p = memCache.get(combinedKey);
    if (!p) {
        p = stats_1.PackageCacheStats.wrapGet(() => cacheProxy.get(namespace, key));
        memCache.set(combinedKey, p);
    }
    const result = await p;
    return result;
}
async function set(namespace, key, value, minutes) {
    if (!cacheProxy) {
        return;
    }
    await stats_1.PackageCacheStats.wrapSet(() => cacheProxy.set(namespace, key, value, minutes));
    const combinedKey = (0, key_1.getCombinedKey)(namespace, key);
    const p = Promise.resolve(value);
    memCache.set(combinedKey, p);
}
async function init(config) {
    if (config.redisUrl) {
        await redisCache.init(config.redisUrl, config.redisPrefix);
        cacheProxy = {
            get: redisCache.get,
            set: redisCache.set,
        };
        return;
    }
    if ((0, env_1.getEnv)().RENOVATE_X_SQLITE_PACKAGE_CACHE) {
        cacheProxy = await sqlite_1.SqlitePackageCache.init(config.cacheDir);
        return;
    }
    if (config.cacheDir) {
        fileCache.init(config.cacheDir);
        cacheProxy = {
            get: fileCache.get,
            set: fileCache.set,
            cleanup: fileCache.cleanup,
        };
        return;
    }
}
async function cleanup(config) {
    if (config?.redisUrl) {
        await redisCache.end();
    }
    if (cacheProxy?.cleanup) {
        await cacheProxy.cleanup();
    }
}
//# sourceMappingURL=index.js.map