"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.api = exports.isValid = exports.isVersion = exports.supportsRanges = exports.urls = exports.displayName = exports.id = void 0;
exports.getSatisfyingVersion = getSatisfyingVersion;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const semver_1 = tslib_1.__importDefault(require("semver"));
const semver_stable_1 = tslib_1.__importDefault(require("semver-stable"));
const regex_1 = require("../../../util/regex");
const semver_2 = require("../semver");
exports.id = 'semver-coerced';
exports.displayName = 'Coerced Semantic Versioning';
exports.urls = ['https://semver.org/'];
exports.supportsRanges = false;
function isStable(version) {
    // matching a version with the semver prefix
    // v1.2.3, 1.2.3, v1.2, 1.2, v1, 1
    const regx = (0, regex_1.regEx)(/^v?(?<major>\d+)(?<minor>\.\d+)?(?<patch>\.\d+)?(?<others>.+)?/);
    const m = regx.exec(version);
    if (!m?.groups) {
        return false;
    }
    const major = m.groups.major;
    const newMinor = m.groups.minor ?? '.0';
    const newPatch = m.groups.patch ?? '.0';
    const others = m.groups.others ?? '';
    const fixed = major + newMinor + newPatch + others;
    return semver_stable_1.default.is(fixed);
}
function sortVersions(a, b) {
    const aCoerced = semver_1.default.coerce(a);
    const bCoerced = semver_1.default.coerce(b);
    return aCoerced && bCoerced ? semver_1.default.compare(aCoerced, bCoerced) : 0;
}
function getMajor(a) {
    const aCoerced = semver_1.default.coerce(a);
    return aCoerced ? semver_1.default.major(aCoerced) : null;
}
function getMinor(a) {
    const aCoerced = semver_1.default.coerce(a);
    return aCoerced ? semver_1.default.minor(aCoerced) : null;
}
function getPatch(a) {
    const aCoerced = semver_1.default.coerce(a);
    return aCoerced ? semver_1.default.patch(aCoerced) : null;
}
function matches(version, range) {
    const coercedVersion = semver_1.default.coerce(version);
    return coercedVersion ? semver_1.default.satisfies(coercedVersion, range) : false;
}
function equals(a, b) {
    const aCoerced = semver_1.default.coerce(a);
    const bCoerced = semver_1.default.coerce(b);
    return aCoerced && bCoerced ? semver_1.default.eq(aCoerced, bCoerced) : false;
}
function isValid(version) {
    return !!semver_1.default.valid(semver_1.default.coerce(version));
}
function getSatisfyingVersion(versions, range) {
    const coercedVersions = versions
        .map((version) => semver_1.default.valid(version) ? version : semver_1.default.coerce(version)?.version)
        .filter(is_1.default.string);
    return semver_1.default.maxSatisfying(coercedVersions, range);
}
function minSatisfyingVersion(versions, range) {
    const coercedVersions = versions
        .map((version) => semver_1.default.coerce(version)?.version)
        .filter(is_1.default.string);
    return semver_1.default.minSatisfying(coercedVersions, range);
}
function isLessThanRange(version, range) {
    const coercedVersion = semver_1.default.coerce(version);
    return coercedVersion ? semver_1.default.ltr(coercedVersion, range) : false;
}
function isGreaterThan(version, other) {
    const coercedVersion = semver_1.default.coerce(version);
    const coercedOther = semver_1.default.coerce(other);
    if (!coercedVersion || !coercedOther) {
        return false;
    }
    return semver_1.default.gt(coercedVersion, coercedOther);
}
const startsWithNumberRegex = (0, regex_1.regEx)(`^\\d`);
function isSingleVersion(version) {
    // Since coercion accepts ranges as well as versions, we have to manually
    // check that the version string starts with either 'v' or a digit.
    if (!version.startsWith('v') && !startsWithNumberRegex.exec(version)) {
        return false;
    }
    return !!semver_1.default.valid(semver_1.default.coerce(version));
}
// If this is left as an alias, inputs like "17.04.0" throw errors
const isVersion = (input) => isValid(input);
exports.isVersion = isVersion;
exports.isValid = exports.isVersion;
function getNewValue({ currentValue, currentVersion, newVersion, }) {
    if (currentVersion === `v${currentValue}`) {
        return newVersion.replace(/^v/, '');
    }
    return newVersion;
}
function isBreaking(version, current) {
    const coercedVersion = semver_1.default.coerce(version)?.toString();
    const coercedCurrent = semver_1.default.coerce(current)?.toString();
    return !!(coercedVersion &&
        coercedCurrent &&
        (0, semver_2.isBreaking)(coercedVersion, coercedCurrent));
}
function isCompatible(version) {
    return (0, exports.isVersion)(version);
}
exports.api = {
    equals,
    getMajor,
    getMinor,
    getPatch,
    isBreaking,
    isCompatible,
    isGreaterThan,
    isLessThanRange,
    isSingleVersion,
    isStable,
    isValid,
    isVersion: exports.isVersion,
    matches,
    getSatisfyingVersion,
    minSatisfyingVersion,
    getNewValue,
    sortVersions,
};
exports.default = exports.api;
//# sourceMappingURL=index.js.map