"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.replacePart = replacePart;
const logger_1 = require("../../../../logger");
const operator_1 = require("../operator");
const range_1 = require("../range");
const version_1 = require("../version");
// Common logic for replace, widen, and bump strategies
// It basically makes the range stick to the new version.
function replacePart(part, to) {
    const { operator, version: ver, companion } = part;
    switch (operator) {
        case operator_1.LT:
            return { ...part, version: (0, version_1.increment)(ver, to) };
        case operator_1.LTE:
            return { ...part, version: to };
        case operator_1.PGTE:
            if (companion) {
                return {
                    ...part,
                    version: (0, version_1.floor)((0, version_1.adapt)(to, ver)),
                    companion: { ...companion, version: to },
                };
            }
            else {
                return { ...part, version: (0, version_1.floor)((0, version_1.adapt)(to, ver)) };
            }
        case operator_1.GT:
            return { ...part, version: (0, version_1.decrement)(to) };
        case operator_1.GTE:
        case operator_1.EQUAL:
            return { ...part, version: to };
        case operator_1.NOT_EQUAL:
            return part;
        // istanbul ignore next
        default:
            logger_1.logger.warn({ operator }, `Unsupported ruby versioning operator`);
            return { operator: '', delimiter: ' ', version: '' };
    }
}
exports.default = ({ range, to }) => {
    const parts = (0, range_1.parseRanges)(range).map((part) => {
        if ((0, range_1.satisfiesRange)(to, part)) {
            // The new version satisfies the range. Keep it as-is.
            // Note that consecutive `~>` and `>=` parts are combined into one Range object,
            // therefore both parts are updated if the new version violates one of them.
            return part;
        }
        if (part.version.split('.').length > to.split('.').length) {
            const diff = part.version.split('.').length - to.split('.').length;
            const versionToReplace = [to, ...Array(diff).fill('0')].join('.');
            const replacement = replacePart(part, versionToReplace);
            return {
                ...replacement,
                version: replacement.version.split('.').slice(0, -diff).join('.'),
            };
        }
        return replacePart(part, to);
    });
    return (0, range_1.stringifyRanges)(parts);
};
//# sourceMappingURL=replace.js.map