"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.api = exports.supportedRangeStrategies = exports.supportsRanges = exports.urls = exports.displayName = exports.id = void 0;
const parser_1 = require("./parser");
const range_1 = require("./range");
const version_1 = require("./version");
exports.id = 'nuget';
exports.displayName = 'NuGet';
exports.urls = [
    'https://docs.microsoft.com/en-us/nuget/concepts/package-versioning',
    'https://nugettools.azurewebsites.net/',
];
exports.supportsRanges = true;
exports.supportedRangeStrategies = ['pin', 'bump'];
class NugetVersioningApi {
    isCompatible(version, _current) {
        return this.isValid(version);
    }
    isSingleVersion(version) {
        const r = (0, parser_1.parseRange)(version);
        if (!r) {
            return false;
        }
        return r.type === 'nuget-exact-range';
    }
    isStable(version) {
        const v = (0, parser_1.parseVersion)(version);
        if (v) {
            return !v.prerelease;
        }
        const r = (0, parser_1.parseRange)(version);
        if (!r || r.type !== 'nuget-exact-range') {
            return false;
        }
        return !r.version.prerelease;
    }
    isValid(input) {
        const v = (0, parser_1.parseVersion)(input);
        if (v) {
            return true;
        }
        const r = (0, parser_1.parseRange)(input);
        if (r) {
            return true;
        }
        return false;
    }
    isVersion(input) {
        if (!input) {
            return false;
        }
        const v = (0, parser_1.parseVersion)(input);
        if (!v) {
            return false;
        }
        return true;
    }
    getMajor(version) {
        const v = (0, parser_1.parseVersion)(version);
        if (!v) {
            return null;
        }
        return v.major;
    }
    getMinor(version) {
        const v = (0, parser_1.parseVersion)(version);
        if (!v) {
            return null;
        }
        return v.minor ?? null;
    }
    getPatch(version) {
        const v = (0, parser_1.parseVersion)(version);
        if (!v) {
            return null;
        }
        return v.patch ?? null;
    }
    equals(version, other) {
        const x = (0, parser_1.parseVersion)(version);
        const y = (0, parser_1.parseVersion)(other);
        if (!x || !y) {
            return false;
        }
        return (0, version_1.compare)(x, y) === 0;
    }
    isGreaterThan(version, other) {
        const x = (0, parser_1.parseVersion)(version);
        const y = (0, parser_1.parseVersion)(other);
        if (!x || !y) {
            return false;
        }
        return (0, version_1.compare)(x, y) > 0;
    }
    isLessThanRange(version, range) {
        const v = (0, parser_1.parseVersion)(version);
        if (!v) {
            return false;
        }
        const u = (0, parser_1.parseVersion)(range);
        if (u) {
            return (0, version_1.compare)(v, u) < 0;
        }
        const r = (0, parser_1.parseRange)(range);
        if (!r) {
            return false;
        }
        if (r.type === 'nuget-exact-range') {
            return (0, version_1.compare)(v, r.version) < 0;
        }
        if (r.type === 'nuget-bracket-range') {
            if (!r.min) {
                return false;
            }
            const minBound = r.min.type === 'nuget-version'
                ? r.min
                : (0, range_1.getFloatingRangeLowerBound)(r.min);
            const cmp = (0, version_1.compare)(v, minBound);
            return r.minInclusive ? cmp < 0 : cmp <= 0;
        }
        const minBound = (0, range_1.getFloatingRangeLowerBound)(r);
        return (0, version_1.compare)(v, minBound) < 0;
    }
    getSatisfyingVersion(versions, range) {
        const r = (0, parser_1.parseRange)(range);
        if (r) {
            let result = null;
            let vMax;
            for (const version of versions) {
                const v = (0, parser_1.parseVersion)(version);
                if (!v) {
                    continue;
                }
                if (!(0, range_1.matches)(v, r)) {
                    continue;
                }
                if (!vMax || (0, version_1.compare)(v, vMax) > 0) {
                    vMax = v;
                    result = version;
                }
            }
            return result;
        }
        const u = (0, parser_1.parseVersion)(range);
        if (u) {
            let result = null;
            let vMax;
            for (const version of versions) {
                const v = (0, parser_1.parseVersion)(version);
                if (!v) {
                    continue;
                }
                if ((0, version_1.compare)(v, u) < 0) {
                    continue;
                }
                if (!vMax || (0, version_1.compare)(v, vMax) > 0) {
                    vMax = v;
                    result = version;
                }
            }
            return result;
        }
        return null;
    }
    minSatisfyingVersion(versions, range) {
        const r = (0, parser_1.parseRange)(range);
        if (r) {
            let result = null;
            let vMin;
            for (const version of versions) {
                const v = (0, parser_1.parseVersion)(version);
                if (!v) {
                    continue;
                }
                if (!(0, range_1.matches)(v, r)) {
                    continue;
                }
                if (!vMin || (0, version_1.compare)(v, vMin) < 0) {
                    result = version;
                    vMin = v;
                }
            }
            return result;
        }
        const u = (0, parser_1.parseVersion)(range);
        if (u) {
            let result = null;
            let vMin;
            for (const version of versions) {
                const v = (0, parser_1.parseVersion)(version);
                if (!v) {
                    continue;
                }
                if ((0, version_1.compare)(v, u) < 0) {
                    continue;
                }
                if (!vMin || (0, version_1.compare)(v, vMin) < 0) {
                    result = version;
                    vMin = v;
                }
            }
            return result;
        }
        return null;
    }
    getNewValue({ currentValue, rangeStrategy, currentVersion, newVersion, }) {
        const v = (0, parser_1.parseVersion)(newVersion);
        if (!v) {
            return null;
        }
        if (rangeStrategy === 'pin') {
            return (0, range_1.rangeToString)({ type: 'nuget-exact-range', version: v });
        }
        if (this.isVersion(currentValue)) {
            return newVersion;
        }
        const r = (0, parser_1.parseRange)(currentValue);
        if (!r) {
            return null;
        }
        if (this.isLessThanRange(newVersion, currentValue)) {
            return currentValue;
        }
        if (r.type === 'nuget-exact-range') {
            return (0, range_1.rangeToString)({ type: 'nuget-exact-range', version: v });
        }
        if (r.type === 'nuget-floating-range') {
            const floating = r.floating;
            if (!floating) {
                return (0, version_1.versionToString)(v);
            }
            const res = { ...r };
            if (floating === 'major') {
                res.major = (0, range_1.coerceFloatingComponent)(v.major);
                return (0, range_1.tryBump)(res, v, currentValue);
            }
            res.major = v.major;
            if (floating === 'minor') {
                res.minor = (0, range_1.coerceFloatingComponent)(v.minor);
                return (0, range_1.tryBump)(res, v, currentValue);
            }
            res.minor = v.minor ?? 0;
            if (floating === 'patch') {
                res.patch = (0, range_1.coerceFloatingComponent)(v.patch);
                return (0, range_1.tryBump)(res, v, currentValue);
            }
            res.patch = v.patch ?? 0;
            res.revision = (0, range_1.coerceFloatingComponent)(v.revision);
            return (0, range_1.tryBump)(res, v, currentValue);
        }
        const res = { ...r };
        if (!r.max) {
            res.min = v;
            res.minInclusive = true;
            return (0, range_1.rangeToString)(res);
        }
        if ((0, range_1.matches)(v, r)) {
            return currentValue;
        }
        if (!r.min) {
            res.max = v;
            res.maxInclusive = true;
            return (0, range_1.rangeToString)(res);
        }
        res.max = v;
        res.maxInclusive = true;
        return (0, range_1.rangeToString)(res);
    }
    sortVersions(version, other) {
        const x = (0, parser_1.parseVersion)(version);
        const y = (0, parser_1.parseVersion)(other);
        if (!x || !y) {
            return 0;
        }
        return (0, version_1.compare)(x, y);
    }
    matches(version, range) {
        const v = (0, parser_1.parseVersion)(version);
        if (!v) {
            return false;
        }
        const u = (0, parser_1.parseVersion)(range);
        if (u) {
            return (0, version_1.compare)(v, u) >= 0;
        }
        const r = (0, parser_1.parseRange)(range);
        if (!r) {
            return false;
        }
        return (0, range_1.matches)(v, r);
    }
}
exports.api = new NugetVersioningApi();
exports.default = exports.api;
//# sourceMappingURL=index.js.map