/**
 * @fileoverview Contains classes that represent a Bazel module version.
 */
/**
 * Represents a single value in a VersionPart. For example, the version string
 * `1.2.3` has three identifiers: `1`, `2`, `3`.
 */
export declare class Identifier {
    /**
     * Returns the identifier as a string.
     */
    readonly asString: string;
    /**
     * If the identifier only contains digits, this is the numeric value.
     * Otherwise, it is `0`.
     */
    readonly asNumber: number;
    /**
     * Specifies whether the identifier only contains digits.
     */
    readonly isDigitsOnly: boolean;
    /**
     * Regular expression used to identify whether an identifier value only
     * contains digits.
     */
    static readonly digitsOnlyMatcher: RegExp;
    /**
     * @param value The value that is parsed for the Bazel module version parts.
     */
    constructor(value: string);
    /**
     * Determines whether this identifier and another identifier are equal.
     */
    equals(other: Identifier): boolean;
    /**
     * Determines whether this identifier comes before the other identifier.
     */
    isLessThan(other: Identifier): boolean;
}
/**
 * A collection of {@link Identifier} values that represent a portion of a
 * Bazel module version.
 */
export declare class VersionPart extends Array<Identifier> {
    /**
     * Creates a {@link VersionPart} populated with the provided identifiers.
     */
    static create(...items: (Identifier | string)[]): VersionPart;
    /**
     * The string representation of the version part.
     */
    get asString(): string;
    /**
     * Specifies whether this contains any identifiers.
     */
    get isEmpty(): boolean;
    /**
     * Returns the equivalent of the a Semver major value.
     */
    get major(): number;
    /**
     * Returns the equivalent of the a Semver minor value.
     */
    get minor(): number;
    /**
     * Returns the equivalent of the a Semver patch value.
     */
    get patch(): number;
    /**
     * Determines whether this version part is equal to the other.
     */
    equals(other: VersionPart): boolean;
    /**
     * Determines whether this version part comes before the other.
     */
    isLessThan(other: VersionPart): boolean;
}
/**
 * Represents a version in the Bazel module system. The version format we support is
 * `RELEASE[-PRERELEASE][+BUILD]`, where `RELEASE`, `PRERELEASE`, and `BUILD` are
 * each a sequence of "identifiers" (defined as a non-empty sequence of ASCII alphanumerical
 * characters and hyphens) separated by dots. The `RELEASE` part may not contain hyphens.
 *
 * Otherwise, this format is identical to SemVer, especially in terms of the comparison algorithm
 * (https://semver.org/#spec-item-11). In other words, this format is intentionally looser than
 * SemVer; in particular:
 *
 * - the "release" part isn't limited to exactly 3 segments (major, minor, patch), but can be
 *   fewer or more;
 * - each segment in the "release" part can be identifiers instead of just numbers (so letters
 *   are also allowed -- although hyphens are not).
 *
 * Any valid SemVer version is a valid Bazel module version. Additionally, two SemVer versions
 * `a` and `b` compare `a < b` iff the same holds when they're compared as Bazel * module versions.
 *
 * The special "empty string" version can also be used, and compares higher than everything else.
 * It signifies that there is a NonRegistryOverride for a module.
 */
export declare class BzlmodVersion {
    readonly original: string;
    readonly release: VersionPart;
    readonly prerelease: VersionPart;
    readonly build: VersionPart;
    /**
     * The regular expression that identifies a valid Bazel module version.
     */
    static readonly versionMatcher: RegExp;
    /**
     * @param version The string that is parsed for the Bazel module version
     *     values.
     */
    constructor(version: string);
    /**
     * Specifies whether this is a pre-release version.
     */
    get isPrerelease(): boolean;
    /**
     * Determines whether this Bazel module version is equal to the other.
     *
     * @param other The other version for the comparison.
     * @param ignoreBuild? If specified, determines whether the build value is
     *     evaluated as part of the equality check. This is useful when
     *     determining precedence.
     */
    equals(other: BzlmodVersion, ignoreBuild?: boolean): boolean;
    /**
     * Determines whether this Bazel module version comes before the other.
     */
    isLessThan(other: BzlmodVersion): boolean;
    /**
     * Determines whether this Bazel module version comes after the other.
     */
    isGreaterThan(other: BzlmodVersion): boolean;
    /**
     * Evaluates two Bazel module versions and returns a value specifying whether
     * a < b (-1), a == b (0), or a > b (1).
     */
    static defaultCompare(a: BzlmodVersion, b: BzlmodVersion): number;
}
