"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.platform = exports.getPlatformList = void 0;
exports.setPlatformApi = setPlatformApi;
exports.initPlatform = initPlatform;
const tslib_1 = require("tslib");
const node_url_1 = tslib_1.__importDefault(require("node:url"));
const error_messages_1 = require("../../constants/error-messages");
const logger_1 = require("../../logger");
const git_1 = require("../../util/git");
const hostRules = tslib_1.__importStar(require("../../util/host-rules"));
const api_1 = tslib_1.__importDefault(require("./api"));
const scm_1 = require("./scm");
const getPlatformList = () => Array.from(api_1.default.keys());
exports.getPlatformList = getPlatformList;
let _platform;
const handler = {
    get(_target, prop) {
        if (!_platform) {
            throw new Error(error_messages_1.PLATFORM_NOT_FOUND);
        }
        return _platform[prop];
    },
};
exports.platform = new Proxy({}, handler);
function setPlatformApi(name) {
    if (!api_1.default.has(name)) {
        throw new Error(`Init: Platform "${name}" not found. Must be one of: ${(0, exports.getPlatformList)().join(', ')}`);
    }
    _platform = api_1.default.get(name);
    (0, scm_1.setPlatformScmApi)(name);
}
async function initPlatform(config) {
    (0, git_1.setPrivateKey)(config.gitPrivateKey);
    (0, git_1.setNoVerify)(config.gitNoVerify ?? []);
    // TODO: `platform` (#22198)
    setPlatformApi(config.platform);
    // TODO: types
    const platformInfo = await exports.platform.initPlatform(config);
    const returnConfig = {
        ...config,
        ...platformInfo,
        hostRules: [
            ...(platformInfo?.hostRules ?? []),
            ...(config.hostRules ?? []),
        ],
    };
    if (config?.gitAuthor) {
        logger_1.logger.debug(`Using configured gitAuthor (${config.gitAuthor})`);
        returnConfig.gitAuthor = config.gitAuthor;
    }
    else if (platformInfo?.gitAuthor) {
        logger_1.logger.debug(`Using platform gitAuthor: ${String(platformInfo.gitAuthor)}`);
        returnConfig.gitAuthor = platformInfo.gitAuthor;
    }
    // This is done for validation and will be overridden later once repo config is incorporated
    (0, git_1.setGitAuthor)(returnConfig.gitAuthor);
    const platformRule = {
        // TODO: null check (#22198)
        matchHost: node_url_1.default.parse(returnConfig.endpoint).hostname,
    };
    // There might have been platform-specific modifications to the token
    if (returnConfig.token) {
        config.token = returnConfig.token;
    }
    ['token', 'username', 'password'].forEach((field) => {
        if (config[field]) {
            // TODO: types #22198
            platformRule[field] = config[field];
            delete returnConfig[field];
        }
    });
    const typedPlatformRule = {
        ...platformRule,
        hostType: returnConfig.platform,
    };
    returnConfig.hostRules.push(typedPlatformRule);
    hostRules.add(typedPlatformRule);
    return returnConfig;
}
//# sourceMappingURL=index.js.map