"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.massageMarkdownLinks = massageMarkdownLinks;
const remark_1 = require("remark");
const logger_1 = require("../../../logger");
const number_1 = require("../../../util/number");
const regex_1 = require("../../../util/regex");
//according to https://github.com/dead-claudia/github-limits
const urlRegex = /(?:https?:)?(?:\/\/)?(?:www\.)?(?<!api\.)(?:to)?github\.com\/[-a-z0-9]+\/[-_a-z0-9.]+\/(?:discussions|issues|pull)\/[0-9]+(?:#[-_a-z0-9]+)?/i; // TODO #12872 (?<!re) after text not matching
function massageLink(input) {
    return input.replace((0, regex_1.regEx)(/(?:to|redirect\.|www\.)?github\.com/i), 'redirect.github.com');
}
function collectLinkPosition(input, matches) {
    const transformer = (tree) => {
        const startOffset = (0, number_1.coerceNumber)(tree.position?.start.offset);
        const endOffset = (0, number_1.coerceNumber)(tree.position?.end.offset);
        if (tree.type === 'link') {
            const substr = input.slice(startOffset, endOffset);
            const url = tree.url;
            const offset = startOffset + substr.lastIndexOf(url);
            if (urlRegex.test(url)) {
                matches.push({
                    start: offset,
                    end: offset + url.length,
                    replaceTo: massageLink(url),
                });
            }
        }
        else if (tree.type === 'text') {
            const globalUrlReg = new RegExp(urlRegex, 'gi');
            const urlMatches = [...tree.value.matchAll(globalUrlReg)];
            for (const match of urlMatches) {
                const [url] = match;
                const start = startOffset + (0, number_1.coerceNumber)(match.index);
                const end = start + url.length;
                const newUrl = massageLink(url);
                matches.push({ start, end, replaceTo: `[${url}](${newUrl})` });
            }
        }
        else if ('children' in tree) {
            tree.children.forEach((child) => {
                transformer(child);
            });
        }
    };
    return () => transformer;
}
function massageMarkdownLinks(content) {
    try {
        const rightSpaces = content.replace(content.trimEnd(), '');
        const matches = [];
        (0, remark_1.remark)().use(collectLinkPosition(content, matches)).processSync(content);
        const result = matches.reduceRight((acc, { start, end, replaceTo }) => {
            const leftPart = acc.slice(0, start);
            const rightPart = acc.slice(end);
            return leftPart + replaceTo + rightPart;
        }, content);
        return result.trimEnd() + rightSpaces;
    }
    catch (err) /* v8 ignore start */ {
        logger_1.logger.warn({ err }, `Unable to massage markdown text`);
        return content;
    } /* v8 ignore stop */
}
//# sourceMappingURL=massage-markdown-links.js.map