"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.REQUEST_DETAILS_FOR_PRS = exports.TAG_PULL_REQUEST_BODY = void 0;
exports.getGerritRepoUrl = getGerritRepoUrl;
exports.mapPrStateToGerritFilter = mapPrStateToGerritFilter;
exports.mapGerritChangeToPr = mapGerritChangeToPr;
exports.mapGerritChangeStateToPrState = mapGerritChangeStateToPrState;
exports.extractSourceBranch = extractSourceBranch;
exports.findPullRequestBody = findPullRequestBody;
exports.mapBranchStatusToLabel = mapBranchStatusToLabel;
const tslib_1 = require("tslib");
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const pr_body_1 = require("../pr-body");
exports.TAG_PULL_REQUEST_BODY = 'pull-request';
exports.REQUEST_DETAILS_FOR_PRS = [
    'MESSAGES', // to get the pr body
    'LABELS', // to get the reviewers
    'DETAILED_ACCOUNTS', // to get the reviewers usernames
    'CURRENT_REVISION', // to get the commit message
    'COMMIT_FOOTERS', // to get the commit message
];
function getGerritRepoUrl(repository, endpoint) {
    // Find options for current host and determine Git endpoint
    const opts = hostRules.find({
        hostType: 'gerrit',
        url: endpoint,
    });
    const url = (0, url_1.parseUrl)(endpoint);
    if (!url) {
        throw new Error(error_messages_1.CONFIG_GIT_URL_UNAVAILABLE);
    }
    if (!(opts.username && opts.password)) {
        throw new Error('Init: You must configure a Gerrit Server username/password');
    }
    url.username = opts.username;
    url.password = opts.password;
    url.pathname = (0, url_1.joinUrlParts)(url.pathname, 'a', encodeURIComponent(repository));
    logger_1.logger.trace({ url: url.toString() }, 'using URL based on configured endpoint');
    return url.toString();
}
function mapPrStateToGerritFilter(state) {
    switch (state) {
        case 'merged':
            return 'status:merged';
        case 'open':
            return 'status:open';
        case 'closed':
            return 'status:abandoned';
        case '!open':
            return '-status:open';
        case 'all':
        default:
            return null;
    }
}
function mapGerritChangeToPr(change, knownProperties) {
    const sourceBranch = knownProperties?.sourceBranch ?? extractSourceBranch(change);
    if (!sourceBranch) {
        return null;
    }
    return {
        number: change._number,
        state: mapGerritChangeStateToPrState(change.status),
        sourceBranch,
        targetBranch: change.branch,
        title: change.subject,
        createdAt: change.created?.replace(' ', 'T'),
        labels: change.hashtags,
        reviewers: change.reviewers?.REVIEWER?.map((reviewer) => reviewer.username) ?? [],
        bodyStruct: {
            hash: (0, pr_body_1.hashBody)(knownProperties?.prBody ?? findPullRequestBody(change)),
        },
        sha: change.current_revision,
    };
}
function mapGerritChangeStateToPrState(state) {
    switch (state) {
        case 'NEW':
            return 'open';
        case 'MERGED':
            return 'merged';
        case 'ABANDONED':
            return 'closed';
    }
}
function extractSourceBranch(change) {
    let sourceBranch = undefined;
    if (change.current_revision) {
        const re = (0, regex_1.regEx)(/^Renovate-Branch: (.+)$/m);
        const currentRevision = change.revisions[change.current_revision];
        const message = currentRevision.commit_with_footers;
        if (message) {
            sourceBranch = re.exec(message)?.[1];
        }
    }
    return sourceBranch ?? undefined;
}
function findPullRequestBody(change) {
    const msg = Array.from(change.messages ?? [])
        .reverse()
        .find((msg) => msg.tag === exports.TAG_PULL_REQUEST_BODY);
    if (msg) {
        return msg.message.replace(/^Patch Set \d+:\n\n/, ''); //TODO: check how to get rid of the auto-added prefix?
    }
    return undefined;
}
function mapBranchStatusToLabel(state, // suppress default path code removal
label) {
    const numbers = Object.keys(label.values).map((x) => parseInt(x, 10));
    switch (state) {
        case 'green':
            return Math.max(...numbers);
        case 'yellow':
        case 'red':
            return Math.min(...numbers);
    }
    /* v8 ignore next */
    return label.default_value;
}
//# sourceMappingURL=utils.js.map