"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRefs = getRefs;
exports.getAzureBranchObj = getAzureBranchObj;
exports.getFile = getFile;
exports.getCommitDetails = getCommitDetails;
exports.getMergeMethod = getMergeMethod;
exports.getAllProjectTeams = getAllProjectTeams;
const tslib_1 = require("tslib");
const GitInterfaces_js_1 = require("azure-devops-node-api/interfaces/GitInterfaces.js");
const logger_1 = require("../../../logger");
const streams_1 = require("../../../util/streams");
const util_1 = require("../util");
const azureApi = tslib_1.__importStar(require("./azure-got-wrapper"));
const schema_1 = require("./schema");
const util_2 = require("./util");
const mergePolicyGuid = 'fa4e907d-c16b-4a4c-9dfa-4916e5d171ab'; // Magic GUID for merge strategy policy configurations
async function getRefs(repoId, branchName) {
    logger_1.logger.debug(`getRefs(${repoId}, ${branchName})`);
    const azureApiGit = await azureApi.gitApi();
    const refs = await azureApiGit.getRefs(repoId, undefined, (0, util_2.getBranchNameWithoutRefsPrefix)(branchName));
    return refs;
}
async function getAzureBranchObj(repoId, branchName, from) {
    const fromBranchName = (0, util_1.getNewBranchName)(from);
    const refs = await getRefs(repoId, fromBranchName);
    if (refs.length === 0) {
        logger_1.logger.debug(`getAzureBranchObj without a valid from, so initial commit.`);
        // TODO: fix undefined
        return {
            name: (0, util_1.getNewBranchName)(branchName),
            oldObjectId: '0000000000000000000000000000000000000000',
        };
    }
    return {
        // TODO: fix undefined (#22198)
        name: (0, util_1.getNewBranchName)(branchName),
        oldObjectId: refs[0].objectId,
    };
}
// if no branchName, look globally
async function getFile(repoId, filePath, branchName) {
    logger_1.logger.trace(`getFile(filePath=${filePath}, branchName=${branchName})`);
    const azureApiGit = await azureApi.gitApi();
    const item = await azureApiGit.getItemText(repoId, filePath, undefined, undefined, 0, // because we look for 1 file
    false, false, true, {
        versionType: 0, // branch
        versionOptions: 0,
        version: (0, util_2.getBranchNameWithoutRefsheadsPrefix)(branchName),
    });
    if (item?.readable) {
        const fileContent = await (0, streams_1.streamToString)(item);
        try {
            const result = schema_1.WrappedExceptionSchema.safeParse(fileContent);
            if (result.success) {
                if (result.data.typeKey === 'GitItemNotFoundException') {
                    logger_1.logger.warn({ filePath }, 'Unable to find file');
                    return null;
                }
                if (result.data.typeKey === 'GitUnresolvableToCommitException') {
                    logger_1.logger.warn({ branchName }, 'Unable to find branch');
                    return null;
                }
            }
        }
        catch /* v8 ignore start */ {
            // it 's not a JSON, so I send the content directly with the line under
        } /* v8 ignore stop */
        return fileContent;
    }
    return null; // no file found
}
async function getCommitDetails(commit, repoId) {
    logger_1.logger.debug(`getCommitDetails(${commit}, ${repoId})`);
    const azureApiGit = await azureApi.gitApi();
    const results = await azureApiGit.getCommit(commit, repoId);
    return results;
}
async function getMergeMethod(repoId, project, branchRef, defaultBranch) {
    logger_1.logger.debug(`getMergeMethod(branchRef=${branchRef}, defaultBranch=${defaultBranch})`);
    const isRelevantScope = (scope) => {
        if (scope.matchKind === 'DefaultBranch' &&
            // TODO: types (#22198)
            (!branchRef || branchRef === `refs/heads/${defaultBranch}`)) {
            return true;
        }
        if (scope.repositoryId !== repoId && scope.repositoryId !== null) {
            return false;
        }
        if (!branchRef) {
            return true;
        }
        // TODO #22198
        return scope.matchKind === 'Exact'
            ? scope.refName === branchRef
            : branchRef.startsWith(scope.refName);
    };
    const policyConfigurations = (await (await azureApi.policyApi()).getPolicyConfigurations(project, undefined, mergePolicyGuid))
        .filter((p) => p.settings.scope.some(isRelevantScope))
        .map((p) => p.settings)[0];
    logger_1.logger.debug(
    // TODO: types (#22198)
    `getMergeMethod(branchRef=${branchRef}) determining mergeMethod from matched policy:\n${JSON.stringify(policyConfigurations, null, 4)}`);
    try {
        // TODO: fix me, wrong types
        return Object.keys(policyConfigurations)
            .map((p) => GitInterfaces_js_1.GitPullRequestMergeStrategy[p.slice(5)])
            .find((p) => p);
    }
    catch {
        return GitInterfaces_js_1.GitPullRequestMergeStrategy.NoFastForward;
    }
}
async function getAllProjectTeams(projectId) {
    const allTeams = [];
    const azureApiCore = await azureApi.coreApi();
    const top = 100;
    let skip = 0;
    let length = 0;
    do {
        const teams = await azureApiCore.getTeams(projectId, undefined, top, skip);
        length = teams.length;
        allTeams.push(...teams);
        skip += top;
    } while (top <= length);
    return allTeams;
}
//# sourceMappingURL=azure-helper.js.map