"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const git_1 = require("../../../util/git");
async function updateArtifacts({ packageFileName, updatedDeps, newPackageFileContent, config, }) {
    logger_1.logger.debug(`vendir.updateArtifacts(${packageFileName})`);
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFileName, 'vendir.lock.yml');
    if (!lockFileName) {
        logger_1.logger.warn('No vendir.lock.yml found');
        return null;
    }
    const existingLockFileContent = await (0, fs_1.readLocalFile)(lockFileName, 'utf8');
    if (!existingLockFileContent) {
        logger_1.logger.warn('Empty vendir.lock.yml found');
        return null;
    }
    try {
        await (0, fs_1.writeLocalFile)(packageFileName, newPackageFileContent);
        logger_1.logger.debug('Updating Vendir artifacts');
        const execOptions = {
            cwdFile: packageFileName,
            docker: {},
            toolConstraints: [
                { toolName: 'vendir', constraint: config.constraints?.vendir },
                { toolName: 'helm', constraint: config.constraints?.helm },
            ],
        };
        await (0, exec_1.exec)(`vendir sync`, execOptions);
        logger_1.logger.debug('Returning updated Vendir artifacts');
        const fileChanges = [];
        const newVendirLockContent = await (0, fs_1.readLocalFile)(lockFileName, 'utf8');
        const isLockFileChanged = existingLockFileContent !== newVendirLockContent;
        if (isLockFileChanged) {
            fileChanges.push({
                file: {
                    type: 'addition',
                    path: lockFileName,
                    contents: newVendirLockContent,
                },
            });
        }
        // add modified vendir archives to artifacts
        logger_1.logger.debug("Adding Sync'd files to git");
        // Files must be in the vendor path to get added
        const vendorDir = (0, fs_1.getParentDir)(packageFileName);
        const status = await (0, git_1.getRepoStatus)();
        if (status) {
            const modifiedFiles = status.modified ?? [];
            const notAddedFiles = status.not_added;
            const deletedFiles = status.deleted ?? [];
            for (const f of modifiedFiles.concat(notAddedFiles)) {
                const isFileInVendorDir = f.startsWith(vendorDir);
                if (vendorDir || isFileInVendorDir) {
                    fileChanges.push({
                        file: {
                            type: 'addition',
                            path: f,
                            contents: await (0, fs_1.readLocalFile)(f),
                        },
                    });
                }
            }
            for (const f of deletedFiles) {
                fileChanges.push({
                    file: {
                        type: 'deletion',
                        path: f,
                    },
                });
            }
        }
        else {
            logger_1.logger.error('Failed to get git status');
        }
        return fileChanges.length ? fileChanges : null;
    }
    catch (err) {
        if (err.message === error_messages_1.TEMPORARY_ERROR) {
            throw err;
        }
        logger_1.logger.debug({ err }, 'Failed to update Vendir lock file');
        return [
            {
                artifactError: {
                    lockFile: lockFileName,
                    stderr: err.message,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map