"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractTFLintPlugin = extractTFLintPlugin;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const github_releases_1 = require("../../datasource/github-releases");
const util_1 = require("./util");
function extractTFLintPlugin(startingLine, lines, pluginName) {
    let lineNumber = startingLine;
    const deps = [];
    let pluginSource = null;
    let currentVersion = null;
    let braceCounter = 0;
    do {
        // istanbul ignore if
        if (lineNumber > lines.length - 1) {
            logger_1.logger.debug(`Malformed TFLint configuration file detected.`);
        }
        const line = lines[lineNumber];
        // istanbul ignore else
        if (is_1.default.string(line)) {
            // `{` will be counted with +1 and `}` with -1.
            // Therefore if we reach braceCounter == 0 then we found the end of the tflint configuration block.
            const openBrackets = (line.match((0, regex_1.regEx)(/\{/g)) ?? []).length;
            const closedBrackets = (line.match((0, regex_1.regEx)(/\}/g)) ?? []).length;
            braceCounter = braceCounter + openBrackets - closedBrackets;
            // only update fields inside the root block
            if (braceCounter === 1) {
                const kvMatch = util_1.keyValueExtractionRegex.exec(line);
                if (kvMatch?.groups) {
                    if (kvMatch.groups.key === 'version') {
                        currentVersion = kvMatch.groups.value;
                    }
                    else if (kvMatch.groups.key === 'source') {
                        pluginSource = kvMatch.groups.value;
                    }
                }
            }
        }
        else {
            // stop - something went wrong
            braceCounter = 0;
        }
        lineNumber += 1;
    } while (braceCounter !== 0);
    const dep = analyseTFLintPlugin(pluginSource, currentVersion);
    deps.push(dep);
    // remove last lineNumber addition to not skip a line after the last bracket
    lineNumber -= 1;
    return { lineNumber, dependencies: deps };
}
function analyseTFLintPlugin(source, version) {
    const dep = {};
    if (source) {
        dep.depType = 'plugin';
        const sourceParts = source.split('/');
        if (sourceParts[0] === 'github.com') {
            dep.currentValue = version;
            dep.datasource = github_releases_1.GithubReleasesDatasource.id;
            dep.depName = sourceParts.slice(1).join('/');
        }
        else {
            dep.skipReason = 'unsupported-datasource';
            dep.depName = source;
        }
    }
    else {
        logger_1.logger.debug({ dep }, 'tflint plugin has no source');
        dep.skipReason = 'no-source';
    }
    return dep;
}
//# sourceMappingURL=plugins.js.map