"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tfrVersionMatchRegex = exports.gitTagsRefMatchRegex = exports.githubRefMatchRegex = void 0;
exports.extractTerragruntModule = extractTerragruntModule;
exports.analyseTerragruntModule = analyseTerragruntModule;
const logger_1 = require("../../../logger");
const common_1 = require("../../../util/common");
const regex_1 = require("../../../util/regex");
const bitbucket_tags_1 = require("../../datasource/bitbucket-tags");
const git_tags_1 = require("../../datasource/git-tags");
const gitea_tags_1 = require("../../datasource/gitea-tags");
const github_tags_1 = require("../../datasource/github-tags");
const gitlab_tags_1 = require("../../datasource/gitlab-tags");
const terraform_module_1 = require("../../datasource/terraform-module");
const providers_1 = require("./providers");
exports.githubRefMatchRegex = (0, regex_1.regEx)(/github\.com([/:])(?<project>[^/]+\/[a-z0-9-_.]+).*\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/i);
exports.gitTagsRefMatchRegex = (0, regex_1.regEx)(/(?:git::)?(?<url>(?:http|https|ssh):\/\/(?:.*@)?(?<host>[^/]*)\/(?<path>.*))\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/);
exports.tfrVersionMatchRegex = (0, regex_1.regEx)(/tfr:\/\/(?<registry>.*?)\/(?<org>[^/]+?)\/(?<name>[^/]+?)\/(?<cloud>[^/?]+).*\?(?:ref|version)=(?<currentValue>.*?)$/);
const hostnameMatchRegex = (0, regex_1.regEx)(/^(?<hostname>[a-zA-Z\d]([a-zA-Z\d-]*\.)+[a-zA-Z\d]+)/);
function extractTerragruntModule(startingLine, lines) {
    const moduleName = 'terragrunt';
    const result = (0, providers_1.extractTerragruntProvider)(startingLine, lines, moduleName);
    result.dependencies.forEach((dep) => {
        // TODO #22198
        dep.managerData.terragruntDependencyType = 'terraform';
    });
    return result;
}
function detectGitTagDatasource(registryUrl) {
    const platform = (0, common_1.detectPlatform)(registryUrl);
    switch (platform) {
        case 'gitlab':
            return gitlab_tags_1.GitlabTagsDatasource.id;
        case 'bitbucket':
            return bitbucket_tags_1.BitbucketTagsDatasource.id;
        case 'gitea':
            return gitea_tags_1.GiteaTagsDatasource.id;
        default:
            return git_tags_1.GitTagsDatasource.id;
    }
}
function analyseTerragruntModule(dep) {
    // TODO #22198
    const source = dep.managerData.source;
    const githubRefMatch = exports.githubRefMatchRegex.exec(source ?? '');
    const gitTagsRefMatch = exports.gitTagsRefMatchRegex.exec(source ?? '');
    const tfrVersionMatch = exports.tfrVersionMatchRegex.exec(source ?? '');
    if (githubRefMatch?.groups) {
        dep.depType = 'github';
        dep.packageName = githubRefMatch.groups.project.replace((0, regex_1.regEx)(/\.git$/), '');
        dep.depName = 'github.com/' + dep.packageName;
        dep.currentValue = githubRefMatch.groups.tag;
        dep.datasource = github_tags_1.GithubTagsDatasource.id;
    }
    else if (gitTagsRefMatch?.groups) {
        const { url, tag } = gitTagsRefMatch.groups;
        const { hostname, host, pathname, protocol } = new URL(url);
        const containsSubDirectory = pathname.includes('//');
        if (containsSubDirectory) {
            logger_1.logger.debug('Terragrunt module contains subdirectory');
        }
        dep.depType = 'gitTags';
        // We don't want to have leading slash, .git or subdirectory in the repository path
        const repositoryPath = pathname
            .replace((0, regex_1.regEx)(/^\//), '')
            .split('//')[0]
            .replace((0, regex_1.regEx)('.git$'), '');
        dep.depName = `${hostname}/${repositoryPath}`;
        dep.currentValue = tag;
        dep.datasource = detectGitTagDatasource(url);
        if (dep.datasource === git_tags_1.GitTagsDatasource.id) {
            if (containsSubDirectory) {
                const tempLookupName = url.split('//');
                dep.packageName = tempLookupName[0] + '//' + tempLookupName[1];
            }
            else {
                dep.packageName = url;
            }
        }
        else {
            // The packageName should only contain the path to the repository
            dep.packageName = repositoryPath;
            dep.registryUrls = [
                protocol === 'https:' ? `https://${host}` : `https://${hostname}`,
            ];
        }
    }
    else if (tfrVersionMatch?.groups) {
        dep.depType = 'terragrunt';
        dep.depName =
            tfrVersionMatch.groups.org +
                '/' +
                tfrVersionMatch.groups.name +
                '/' +
                tfrVersionMatch.groups.cloud;
        dep.currentValue = tfrVersionMatch.groups.currentValue;
        dep.datasource = terraform_module_1.TerraformModuleDatasource.id;
        if (tfrVersionMatch.groups.registry) {
            dep.registryUrls = [`https://${tfrVersionMatch.groups.registry}`];
        }
    }
    else if (source) {
        const moduleParts = source.split('//')[0].split('/');
        if (moduleParts[0] === '..') {
            dep.skipReason = 'local';
        }
        else if (moduleParts.length >= 3) {
            const hostnameMatch = hostnameMatchRegex.exec(source);
            if (hostnameMatch?.groups) {
                dep.registryUrls = [`https://${hostnameMatch.groups.hostname}`];
            }
            dep.depType = 'terragrunt';
            dep.depName = moduleParts.join('/');
            dep.datasource = terraform_module_1.TerraformModuleDatasource.id;
        }
    }
    else {
        logger_1.logger.debug({ dep }, 'terragrunt dep has no source');
        dep.skipReason = 'no-source';
    }
}
//# sourceMappingURL=modules.js.map