"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GenericDockerImageRefExtractor = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const extract_1 = require("../../../dockerfile/extract");
const base_1 = require("../../base");
const utils_1 = require("./utils");
class GenericDockerImageRefExtractor extends base_1.DependencyExtractor {
    getCheckList() {
        return utils_1.generic_image_resource.map((value) => `"${value.type}"`);
    }
    extract(hclMap, _locks, config) {
        const dependencies = [];
        dependencies.push(...this.extractResources(hclMap.resource, utils_1.generic_image_resource, config));
        dependencies.push(...this.extractResources(hclMap.data, utils_1.generic_image_datasource, config));
        return dependencies;
    }
    extractResources(typeMap, image_definitions, config) {
        if (is_1.default.nullOrUndefined(typeMap)) {
            return [];
        }
        const dependencies = [];
        for (const image_resource_def of image_definitions) {
            const { type, path } = image_resource_def;
            const resourceInstancesMap = typeMap[type];
            // is there a resource with current looked at type ( `image_resource_def` )
            if (!is_1.default.nonEmptyObject(resourceInstancesMap)) {
                continue;
            }
            // loop over instances of a resource type
            for (const instance of Object.values(resourceInstancesMap).flat()) {
                dependencies.push(...this.walkPath({ depType: type }, instance, path, config));
            }
        }
        return dependencies;
    }
    /**
     * Recursively follow the path to find elements on the path.
     * If a path element is '*' the parentElement will be interpreted as a list
     * and each element will be followed
     * @param abstractDep dependency which will used as basis for adding attributes
     * @param parentElement element from which the next element will be extracted
     * @param leftPath path elements left to walk down
     */
    walkPath(abstractDep, parentElement, leftPath, config) {
        const dependencies = [];
        // if there are no path elements left, we have reached the end of the path
        if (leftPath.length === 0) {
            // istanbul ignore if
            if (!is_1.default.nonEmptyString(parentElement)) {
                return [
                    {
                        ...abstractDep,
                        skipReason: 'invalid-dependency-specification',
                    },
                ];
            }
            const test = (0, extract_1.getDep)(parentElement, true, config.registryAliases);
            const dep = {
                ...abstractDep,
                ...test,
            };
            return [dep];
        }
        // is this a list iterator
        const pathElement = leftPath[0];
        // get sub element
        const element = is_1.default.nonEmptyObject(parentElement)
            ? parentElement[pathElement]
            : null;
        if (is_1.default.nullOrUndefined(element)) {
            return leftPath.length === 1 // if this is the last element assume a false defined dependency
                ? [
                    {
                        ...abstractDep,
                        skipReason: 'invalid-dependency-specification',
                    },
                ]
                : [];
        }
        if (is_1.default.array(element)) {
            for (const arrayElement of element) {
                dependencies.push(...this.walkPath(abstractDep, arrayElement, leftPath.slice(1), config));
            }
            return dependencies;
        }
        return this.walkPath(abstractDep, element, leftPath.slice(1), config);
    }
}
exports.GenericDockerImageRefExtractor = GenericDockerImageRefExtractor;
//# sourceMappingURL=generic-docker-image-ref.js.map