"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModuleExtractor = exports.hostnameMatchRegex = exports.azureDevOpsSshRefMatchRegex = exports.gitTagsRefMatchRegex = exports.bitbucketRefMatchRegex = exports.githubRefMatchRegex = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../../../logger");
const regex_1 = require("../../../../../util/regex");
const bitbucket_tags_1 = require("../../../../datasource/bitbucket-tags");
const git_tags_1 = require("../../../../datasource/git-tags");
const github_tags_1 = require("../../../../datasource/github-tags");
const terraform_module_1 = require("../../../../datasource/terraform-module");
const base_1 = require("../../base");
exports.githubRefMatchRegex = (0, regex_1.regEx)(/github\.com([/:])(?<project>[^/]+\/[a-z0-9-_.]+).*\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/i);
exports.bitbucketRefMatchRegex = (0, regex_1.regEx)(/(?:git::)?(?<url>(?:http|https|ssh)?(?::\/\/)?(?:.*@)?(?<path>bitbucket\.org\/(?<workspace>.*)\/(?<project>.*)\.git\/?(?<subfolder>.*)))\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/);
exports.gitTagsRefMatchRegex = (0, regex_1.regEx)(/(?:git::)?(?<url>(?:(?:http|https|ssh):\/\/)?(?:.*@)?(?<path>[^:/]+[:/](?<project>[^/]+(?:\/[^/]+)*))(?:\.git)?)((\/\/)?(?<subfolder>[^?]*))?\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/);
exports.azureDevOpsSshRefMatchRegex = (0, regex_1.regEx)(/(?:git::)?(?<url>git@ssh\.dev\.azure\.com:v3\/(?<organization>[^/]*)\/(?<project>[^/]*)\/(?<repository>[^/]*))(?<modulepath>.*)?\?(depth=\d+&)?ref=(?<tag>.*?)(&depth=\d+)?$/);
exports.hostnameMatchRegex = (0, regex_1.regEx)(/^(?<hostname>[a-zA-Z\d]([a-zA-Z\d-]*\.)+[a-zA-Z\d]+)/);
class ModuleExtractor extends base_1.DependencyExtractor {
    getCheckList() {
        return ['module'];
    }
    extract(hclRoot) {
        const modules = hclRoot.module;
        if (is_1.default.nullOrUndefined(modules)) {
            return [];
        }
        // istanbul ignore if
        if (!is_1.default.plainObject(modules)) {
            logger_1.logger.debug({ modules }, 'Terraform: unexpected `modules` value');
            return [];
        }
        const dependencies = [];
        for (const [depName, moduleElements] of Object.entries(modules)) {
            for (const moduleElement of moduleElements) {
                const dep = {
                    depName,
                    depType: 'module',
                    currentValue: moduleElement.version,
                    managerData: {
                        source: moduleElement.source,
                    },
                };
                dependencies.push(this.analyseTerraformModule(dep));
            }
        }
        return dependencies;
    }
    analyseTerraformModule(dep) {
        // TODO #22198
        const source = dep.managerData.source;
        const githubRefMatch = exports.githubRefMatchRegex.exec(source);
        const bitbucketRefMatch = exports.bitbucketRefMatchRegex.exec(source);
        const gitTagsRefMatch = exports.gitTagsRefMatchRegex.exec(source);
        const azureDevOpsSshRefMatch = exports.azureDevOpsSshRefMatchRegex.exec(source);
        if (githubRefMatch?.groups) {
            dep.packageName = githubRefMatch.groups.project.replace((0, regex_1.regEx)(/\.git$/), '');
            dep.depName = 'github.com/' + dep.packageName;
            dep.currentValue = githubRefMatch.groups.tag;
            dep.datasource = github_tags_1.GithubTagsDatasource.id;
        }
        else if (bitbucketRefMatch?.groups) {
            dep.depName =
                bitbucketRefMatch.groups.workspace +
                    '/' +
                    bitbucketRefMatch.groups.project;
            dep.packageName = dep.depName;
            dep.currentValue = bitbucketRefMatch.groups.tag;
            dep.datasource = bitbucket_tags_1.BitbucketTagsDatasource.id;
        }
        else if (azureDevOpsSshRefMatch?.groups) {
            dep.depName = `${azureDevOpsSshRefMatch.groups.organization}/${azureDevOpsSshRefMatch.groups.project}/${azureDevOpsSshRefMatch.groups.repository}${azureDevOpsSshRefMatch.groups.modulepath}`;
            dep.packageName = azureDevOpsSshRefMatch.groups.url;
            dep.currentValue = azureDevOpsSshRefMatch.groups.tag;
            dep.datasource = git_tags_1.GitTagsDatasource.id;
        }
        else if (gitTagsRefMatch?.groups) {
            if (gitTagsRefMatch.groups.subfolder) {
                logger_1.logger.debug('Terraform module contains subdirectory');
            }
            dep.depName = gitTagsRefMatch.groups.path.replace('.git', '');
            dep.packageName = gitTagsRefMatch.groups.url.replace('.git', '');
            dep.currentValue = gitTagsRefMatch.groups.tag;
            dep.datasource = git_tags_1.GitTagsDatasource.id;
        }
        else if (source) {
            const moduleParts = source.split('//')[0].split('/');
            if (moduleParts[0] === '.' || moduleParts[0] === '..') {
                dep.skipReason = 'local';
            }
            else if (moduleParts.length >= 3) {
                const hostnameMatch = exports.hostnameMatchRegex.exec(source);
                if (hostnameMatch?.groups) {
                    dep.registryUrls = [`https://${hostnameMatch.groups.hostname}`];
                }
                dep.depName = moduleParts.join('/');
                dep.datasource = terraform_module_1.TerraformModuleDatasource.id;
            }
        }
        else {
            logger_1.logger.debug({ dep }, 'terraform dep has no source');
            dep.skipReason = 'no-source';
        }
        return dep;
    }
}
exports.ModuleExtractor = ModuleExtractor;
//# sourceMappingURL=modules.js.map