"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const regex_1 = require("../../../util/regex");
const yaml_1 = require("../../../util/yaml");
const git_tags_1 = require("../../datasource/git-tags");
const github_releases_1 = require("../../datasource/github-releases");
const extract_1 = require("../dockerfile/extract");
function extractPackageFile(content, packageFile) {
    logger_1.logger.trace(`tekton.extractPackageFile(${packageFile})`);
    const deps = [];
    let docs;
    try {
        // TODO: use schema (#9610)
        docs = (0, yaml_1.parseYaml)(content);
    }
    catch (err) {
        logger_1.logger.debug({ err, packageFile }, 'Failed to parse YAML resource as a Tekton resource');
        return null;
    }
    for (const doc of docs) {
        deps.push(...getDeps(doc));
    }
    if (!deps.length) {
        return null;
    }
    return { deps };
}
function getDeps(doc) {
    const deps = [];
    if (is_1.default.falsy(doc)) {
        return deps;
    }
    // Handle TaskRun resource
    addDep(doc.spec?.taskRef, deps);
    addStepImageSpec(doc.spec?.taskSpec, deps);
    // Handle Task resource
    addStepImageSpec(doc.spec, deps);
    // Handle PipelineRun resource
    addDep(doc.spec?.pipelineRef, deps);
    addPipelineAsCodeAnnotations(doc.metadata?.annotations, deps);
    // Handle PipelineRun resource with inline Pipeline definition
    const pipelineSpec = doc.spec?.pipelineSpec;
    if (is_1.default.truthy(pipelineSpec)) {
        deps.push(...getDeps({ spec: pipelineSpec }));
    }
    // Handle regular tasks of Pipeline resource
    for (const task of [
        ...(0, array_1.coerceArray)(doc.spec?.tasks),
        ...(0, array_1.coerceArray)(doc.spec?.finally),
    ]) {
        addDep(task.taskRef, deps);
        addStepImageSpec(task.taskSpec, deps);
    }
    // Handle TriggerTemplate resource
    for (const resource of (0, array_1.coerceArray)(doc.spec?.resourcetemplates)) {
        deps.push(...getDeps(resource));
    }
    // Handle list of TektonResources
    for (const item of (0, array_1.coerceArray)(doc.items)) {
        deps.push(...getDeps(item));
    }
    return deps;
}
const annotationRegex = (0, regex_1.regEx)(/^pipelinesascode\.tekton\.dev\/(?:task(-[0-9]+)?|pipeline)$/);
function addPipelineAsCodeAnnotations(annotations, deps) {
    if (is_1.default.nullOrUndefined(annotations)) {
        return;
    }
    for (const [key, value] of Object.entries(annotations)) {
        if (!annotationRegex.test(key)) {
            continue;
        }
        const values = value
            .replace((0, regex_1.regEx)(/]$/), '')
            .replace((0, regex_1.regEx)(/^\[/), '')
            .split(',');
        for (const value of values) {
            const dep = getAnnotationDep(value.trim());
            if (!dep) {
                continue;
            }
            deps.push(dep);
        }
    }
}
const githubRelease = (0, regex_1.regEx)(/^(?<url>(?:(?:http|https):\/\/)?(?<path>(?:[^:/\s]+[:/])?(?<project>[^/\s]+\/[^/\s]+)))\/releases\/download\/(?<currentValue>.+)\/(?<subdir>[^?\s]*)$/);
const gitUrl = (0, regex_1.regEx)(/^(?<url>(?:(?:http|https):\/\/)?(?<path>(?:[^:/\s]+[:/])?(?<project>[^/\s]+\/[^/\s]+)))(?:\/raw)?\/(?<currentValue>.+?)\/(?<subdir>[^?\s]*)$/);
function getAnnotationDep(url) {
    const dep = {};
    dep.depType = 'tekton-annotation';
    let groups = githubRelease.exec(url)?.groups;
    if (groups) {
        dep.datasource = github_releases_1.GithubReleasesDatasource.id;
        dep.depName = groups.path;
        dep.packageName = groups.project;
        dep.currentValue = groups.currentValue;
        return dep;
    }
    groups = gitUrl.exec(url)?.groups;
    if (groups) {
        dep.datasource = git_tags_1.GitTagsDatasource.id;
        dep.depName = groups.path.replace('raw.githubusercontent.com', 'github.com');
        dep.packageName = groups.url.replace('raw.githubusercontent.com', 'github.com');
        dep.currentValue = groups.currentValue;
        return dep;
    }
    return null;
}
function addDep(ref, deps) {
    if (is_1.default.falsy(ref)) {
        return;
    }
    let imageRef;
    // First, find a bundle reference from the Bundle resolver
    if (ref.resolver === 'bundles') {
        imageRef = getBundleValue(ref.params);
        if (is_1.default.nullOrUndefined(imageRef)) {
            // Fallback to the deprecated Bundle resolver attribute
            imageRef = getBundleValue(ref.resource);
        }
    }
    if (is_1.default.nullOrUndefined(imageRef)) {
        // Fallback to older style bundle reference
        imageRef = ref.bundle;
    }
    const dep = (0, extract_1.getDep)(imageRef);
    dep.depType = 'tekton-bundle';
    logger_1.logger.trace({
        depName: dep.depName,
        currentValue: dep.currentValue,
        currentDigest: dep.currentDigest,
    }, 'Tekton bundle dependency found');
    deps.push(dep);
}
function addStepImageSpec(spec, deps) {
    if (is_1.default.nullOrUndefined(spec)) {
        return;
    }
    const steps = [
        ...(0, array_1.coerceArray)(spec.steps),
        ...(0, array_1.coerceArray)(spec.sidecars),
        spec.stepTemplate,
    ];
    for (const step of steps) {
        if (is_1.default.nullOrUndefined(step?.image)) {
            continue;
        }
        const dep = (0, extract_1.getDep)(step?.image);
        dep.depType = 'tekton-step-image';
        logger_1.logger.trace({
            depName: dep.depName,
            currentValue: dep.currentValue,
            currentDigest: dep.currentDigest,
        }, 'Tekton step image dependency found');
        deps.push(dep);
    }
}
function getBundleValue(fields) {
    for (const field of (0, array_1.coerceArray)(fields)) {
        if (field.name === 'bundle') {
            return field.value;
        }
    }
    return undefined;
}
//# sourceMappingURL=extract.js.map