"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Puppetfile = void 0;
exports.parsePuppetfile = parsePuppetfile;
const array_1 = require("../../../util/array");
const regex_1 = require("../../../util/regex");
const forgeRegex = (0, regex_1.regEx)(/^forge\s+['"]([^'"]+)['"]/);
const commentRegex = (0, regex_1.regEx)(/#.*$/);
/**
 * For us a Puppetfile is build up of forges that have Modules.
 *
 * Modules are the updatable parts.
 *
 */
class Puppetfile {
    forgeModules = new Map();
    add(currentForge, module) {
        if (Object.keys(module).length === 0) {
            return;
        }
        if (!this.forgeModules.has(currentForge)) {
            this.forgeModules.set(currentForge, []);
        }
        this.forgeModules.get(currentForge)?.push(module);
    }
    getForges() {
        return Array.from(this.forgeModules.keys());
    }
    getModulesOfForge(forgeUrl) {
        const modules = this.forgeModules.get(forgeUrl ?? null);
        return (0, array_1.coerceArray)(modules);
    }
}
exports.Puppetfile = Puppetfile;
function parsePuppetfile(content) {
    const puppetfile = new Puppetfile();
    let currentForge = null;
    let currentPuppetfileModule = {};
    for (const rawLine of content.split(regex_1.newlineRegex)) {
        // remove comments
        const line = rawLine.replace(commentRegex, '');
        const forgeResult = forgeRegex.exec(line);
        if (forgeResult) {
            puppetfile.add(currentForge, currentPuppetfileModule);
            currentPuppetfileModule = {};
            currentForge = forgeResult[1];
            continue;
        }
        const moduleStart = line.startsWith('mod');
        if (moduleStart) {
            puppetfile.add(currentForge, currentPuppetfileModule);
            currentPuppetfileModule = {};
        }
        const moduleValueRegex = (0, regex_1.regEx)(/(?:\s*:(\w+)\s+=>\s+)?['"]([^'"]+)['"]/g);
        let moduleValue;
        while ((moduleValue = moduleValueRegex.exec(line)) !== null) {
            const key = moduleValue[1];
            const value = moduleValue[2];
            if (key) {
                currentPuppetfileModule.tags =
                    currentPuppetfileModule.tags ?? new Map();
                currentPuppetfileModule.tags.set(key, value);
            }
            else {
                fillPuppetfileModule(currentPuppetfileModule, value);
            }
        }
    }
    puppetfile.add(currentForge, currentPuppetfileModule);
    return puppetfile;
}
function fillPuppetfileModule(currentPuppetfileModule, value) {
    // "positional" module values
    if (currentPuppetfileModule.name === undefined) {
        // moduleName
        currentPuppetfileModule.name = value;
    }
    else if (currentPuppetfileModule.version === undefined) {
        // second value without a key is the version
        currentPuppetfileModule.version = value;
    }
    else {
        // 3+ value without a key is not supported
        currentPuppetfileModule.skipReason = 'invalid-config';
    }
}
//# sourceMappingURL=puppetfile-parser.js.map