"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockfileYaml = exports.PixiToml = exports.PixiConfigSchema = void 0;
const tslib_1 = require("tslib");
const zod_1 = require("zod");
const schema_utils_1 = require("../../../util/schema-utils");
const conda_1 = require("../../datasource/conda/");
const git_refs_1 = require("../../datasource/git-refs");
const pypi_1 = require("../../datasource/pypi");
const condaVersion = tslib_1.__importStar(require("../../versioning/conda/"));
const git_1 = require("../../versioning/git");
const pep440_1 = require("../../versioning/pep440/");
const Channel = zod_1.z.union([
    zod_1.z.string(),
    zod_1.z.object({ channel: zod_1.z.string(), priority: zod_1.z.number() }),
]);
function collectNamedPackages(packages) {
    return Object.entries(packages).map(([depName, config]) => {
        return {
            ...config,
            depName,
        };
    });
}
const PypiDependency = zod_1.z
    .union([
    zod_1.z.string().transform((version) => ({ version })),
    zod_1.z.object({ version: zod_1.z.string() }),
])
    .transform(({ version }) => {
    return {
        currentValue: version,
        versioning: pep440_1.id,
        datasource: pypi_1.PypiDatasource.id,
    };
});
const PypiGitDependency = zod_1.z
    .object({ git: zod_1.z.string(), rev: zod_1.z.optional(zod_1.z.string()) })
    .transform(({ git, rev }) => {
    // empty ref default to HEAD, so do we not need to do anything
    if (!rev) {
        return {
            currentValue: rev,
            packageName: git,
            datasource: git_refs_1.GitRefsDatasource.id,
            versioning: git_1.id,
            skipStage: 'extract',
            skipReason: 'unspecified-version',
        };
    }
    return {
        currentValue: rev,
        packageName: git,
        datasource: git_refs_1.GitRefsDatasource.id,
        versioning: git_1.id,
    };
});
const PypiDependencies = (0, schema_utils_1.LooseRecord)(zod_1.z.string(), zod_1.z.union([PypiDependency, PypiGitDependency])).transform(collectNamedPackages);
const CondaDependency = zod_1.z
    .union([
    zod_1.z.string().transform((version) => ({ version, channel: undefined })),
    zod_1.z.object({ version: zod_1.z.string(), channel: zod_1.z.optional(zod_1.z.string()) }),
])
    .transform(({ version, channel }) => {
    return {
        currentValue: version,
        versioning: condaVersion.id,
        datasource: conda_1.CondaDatasource.id,
        channel,
    };
});
const CondaDependencies = (0, schema_utils_1.LooseRecord)(zod_1.z.string(), CondaDependency).transform(collectNamedPackages);
const Targets = (0, schema_utils_1.LooseRecord)(zod_1.z.string(), zod_1.z.object({
    dependencies: zod_1.z.optional(CondaDependencies).default({}),
    'pypi-dependencies': zod_1.z.optional(PypiDependencies).default({}),
})).transform((val) => {
    const conda = [];
    const pypi = [];
    for (const value of Object.values(val)) {
        pypi.push(...value['pypi-dependencies']);
        conda.push(...value.dependencies);
    }
    return { pypi, conda };
});
const Project = zod_1.z.object({
    channels: zod_1.z.array(Channel).default([]),
    'requires-pixi': zod_1.z.string().optional(),
    'channel-priority': zod_1.z
        .union([zod_1.z.literal('strict'), zod_1.z.literal('disabled')])
        .default('strict'),
});
const DependenciesMixin = zod_1.z
    .object({
    dependencies: zod_1.z.optional(CondaDependencies).default({}),
    'pypi-dependencies': zod_1.z.optional(PypiDependencies).default({}),
    target: zod_1.z.optional(Targets).default({}),
})
    .transform((val) => {
    return {
        conda: [...val.dependencies, ...val.target.conda],
        pypi: [...val['pypi-dependencies'], ...val.target.pypi],
    };
});
const Features = (0, schema_utils_1.LooseRecord)(zod_1.z.string(), zod_1.z
    .object({
    channels: zod_1.z.array(Channel).optional(),
})
    .and(DependenciesMixin)).transform((features) => {
    const pypi = [];
    const conda = [];
    for (const [name, feature] of Object.entries(features)) {
        conda.push(...feature.conda.map((item) => {
            return {
                ...item,
                depType: `feature-${name}`,
                channels: feature.channels,
            };
        }));
        pypi.push(...feature.pypi.map((item) => ({
            depType: `feature-${name}`,
            ...item,
        })));
    }
    return { pypi, conda };
});
const PixiWorkspace = zod_1.z
    .object({
    workspace: Project,
})
    .transform((val) => {
    return { project: val.workspace };
});
const PixiProject = zod_1.z.object({
    project: Project,
});
/**
 * `$` of `pixi.toml` or `$.tool.pixi` of `pyproject.toml`
 */
exports.PixiConfigSchema = zod_1.z
    .union([PixiWorkspace, PixiProject])
    .and(zod_1.z.object({ feature: Features.default({}) }))
    .and(DependenciesMixin);
exports.PixiToml = schema_utils_1.Toml.pipe(exports.PixiConfigSchema);
exports.LockfileYaml = schema_utils_1.Yaml.pipe(zod_1.z.object({
    version: zod_1.z.number(),
}));
//# sourceMappingURL=schema.js.map