"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUserPixiConfig = getUserPixiConfig;
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const zod_1 = require("zod");
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const fs_1 = require("../../../util/fs");
const result_1 = require("../../../util/result");
const schema_utils_1 = require("../../../util/schema-utils");
const url_1 = require("../../../util/url");
const common_1 = require("../../datasource/conda/common");
const schema_1 = require("../pep621/schema");
const schema_2 = require("./schema");
const PyProjectToml = schema_utils_1.Toml.pipe(schema_1.PyProjectSchema);
function getUserPixiConfig(content, packageFile) {
    if (packageFile === 'pyproject.toml' ||
        packageFile.endsWith('/pyproject.toml')) {
        const { val, err } = result_1.Result.parse(content, PyProjectToml).unwrap();
        if (err) {
            logger_1.logger.debug({ packageFile, err }, `error parsing ${packageFile}`);
            return null;
        }
        return val.tool?.pixi ?? null;
    }
    if (packageFile === 'pixi.toml' || packageFile.endsWith('/pixi.toml')) {
        const { val, err } = result_1.Result.parse(content, schema_2.PixiToml).unwrap();
        if (err) {
            logger_1.logger.debug({ packageFile, err }, `error parsing ${packageFile}`);
            return null;
        }
        return val;
    }
    const { val, err } = result_1.Result.parse(content, zod_1.z.union([schema_2.PixiToml, PyProjectToml.transform((p) => p.tool?.pixi)])).unwrap();
    if (err) {
        logger_1.logger.debug({ packageFile, err }, `error parsing ${packageFile}`);
        return null;
    }
    return val ?? null;
}
async function extractPackageFile(content, packageFile) {
    logger_1.logger.trace(`pixi.extractPackageFile(${packageFile})`);
    const val = getUserPixiConfig(content, packageFile);
    if (!val) {
        return null;
    }
    const lockfileName = (0, fs_1.getSiblingFileName)(packageFile, 'pixi.lock');
    const lockFiles = [];
    if (await (0, fs_1.localPathExists)(lockfileName)) {
        lockFiles.push(lockfileName);
    }
    const project = val.project;
    const channelPriority = project['channel-priority'];
    let registryStrategy;
    if (channelPriority === 'disabled') {
        registryStrategy = 'merge';
    }
    // resolve channels and build registry urls for each channel with order
    const conda = [];
    for (const item of val.conda) {
        conda.push(addRegistryUrls({
            ...item,
            channels: project.channels,
            registryStrategy,
        }));
    }
    for (const item of val.feature.conda) {
        conda.push(addRegistryUrls({
            ...item,
            registryStrategy,
            channels: [...(0, array_1.coerceArray)(item.channels), ...project.channels],
        }));
    }
    return {
        lockFiles,
        deps: [conda, val.pypi, val.feature.pypi].flat(),
    };
}
function addRegistryUrls(item) {
    const channels = orderChannels(item.channels);
    if (item.channel) {
        return {
            ...item,
            channels,
            registryUrls: [channelToRegistryUrl(item.channel)],
        };
    }
    if (channels.length === 0) {
        return {
            ...item,
            channels,
            skipStage: 'extract',
            skipReason: 'unknown-registry',
        };
    }
    const registryUrls = [];
    for (const channel of channels) {
        registryUrls.push(channelToRegistryUrl(channel));
    }
    return {
        ...item,
        channels,
        registryUrls,
    };
}
function channelToRegistryUrl(channel) {
    if ((0, url_1.isHttpUrl)(channel)) {
        return (0, url_1.ensureTrailingSlash)(channel);
    }
    return (0, url_1.ensureTrailingSlash)((0, url_1.joinUrlParts)(common_1.defaultRegistryUrl, channel));
}
function orderChannels(channels = []) {
    return channels
        .map((channel, index) => {
        if (is_1.default.string(channel)) {
            return { channel, priority: 0, index };
        }
        return { ...channel, index: 0 };
    })
        .toSorted((a, b) => {
        // first based on priority then based on index
        if (a.priority !== b.priority) {
            return b.priority - a.priority;
        }
        return a.index - b.index;
    })
        .map((c) => c.channel);
}
//# sourceMappingURL=extract.js.map