"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.commandLock = void 0;
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const extract_1 = require("./extract");
exports.commandLock = 'pixi lock --no-progress --color=never --quiet';
async function updateArtifacts({ packageFileName, updatedDeps, newPackageFileContent, config, }) {
    logger_1.logger.debug(`pixi.updateArtifacts(${packageFileName})`);
    const { isLockFileMaintenance } = config;
    if (!is_1.default.nonEmptyArray(updatedDeps) && !isLockFileMaintenance) {
        logger_1.logger.debug('No updated pixi deps - returning null');
        return null;
    }
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFileName, 'pixi.lock');
    const existingLockFileContent = await (0, fs_1.readLocalFile)(lockFileName, 'utf8');
    if (!existingLockFileContent) {
        logger_1.logger.debug(`No lock file found`);
        return null;
    }
    logger_1.logger.trace(`Updating ${lockFileName}`);
    const cmd = [exports.commandLock];
    const pixiConfig = (0, extract_1.getUserPixiConfig)(newPackageFileContent, packageFileName);
    const constraint = config.constraints?.pixi ?? pixiConfig?.project['requires-pixi'];
    try {
        await (0, fs_1.writeLocalFile)(packageFileName, newPackageFileContent);
        if (isLockFileMaintenance) {
            await (0, fs_1.deleteLocalFile)(lockFileName);
        }
        // https://pixi.sh/latest/features/environment/#caching-packages
        const PIXI_CACHE_DIR = await (0, fs_1.ensureCacheDir)('pixi');
        const extraEnv = {
            PIXI_CACHE_DIR,
            RATTLER_CACHE_DIR: PIXI_CACHE_DIR,
        };
        const execOptions = {
            cwdFile: packageFileName,
            extraEnv,
            docker: {},
            toolConstraints: [{ toolName: 'pixi', constraint }],
        };
        await (0, exec_1.exec)(cmd, execOptions);
        const newPixiLockContent = await (0, fs_1.readLocalFile)(lockFileName, 'utf8');
        if (existingLockFileContent === newPixiLockContent) {
            logger_1.logger.debug(`${lockFileName} is unchanged`);
            return null;
        }
        return [
            {
                file: {
                    type: 'addition',
                    path: lockFileName,
                    contents: newPixiLockContent,
                },
            },
        ];
    }
    catch (err) {
        if (err.message === error_messages_1.TEMPORARY_ERROR) {
            throw err;
        }
        logger_1.logger.debug({ err }, `Failed to update ${lockFileName} file`);
        return [
            {
                artifactError: {
                    lockFile: lockFileName,
                    stderr: `${err}`,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map