"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.constructPipCompileCmd = constructPipCompileCmd;
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const shlex_1 = require("shlex");
const upath_1 = tslib_1.__importDefault(require("upath"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const env_1 = require("../../../util/env");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const git_1 = require("../../../util/git");
const common_1 = require("../pip_requirements/common");
const common_2 = require("./common");
const utils_1 = require("./utils");
function haveCredentialsInPipEnvironmentVariables() {
    const env = (0, env_1.getEnv)();
    if (env.PIP_INDEX_URL) {
        try {
            const indexUrl = new URL(env.PIP_INDEX_URL);
            if (!!indexUrl.username || !!indexUrl.password) {
                return true;
            }
        }
        catch {
            // Assume that an invalid URL contains credentials, just in case
            return true;
        }
    }
    try {
        if (env.PIP_EXTRA_INDEX_URL) {
            return env.PIP_EXTRA_INDEX_URL.split(' ')
                .map((urlString) => new URL(urlString))
                .some((url) => !!url.username || !!url.password);
        }
    }
    catch {
        // Assume that an invalid URL contains credentials, just in case
        return true;
    }
    return false;
}
function constructPipCompileCmd(compileArgs, upgradePackages = []) {
    if (compileArgs.commandType === 'custom') {
        throw new Error('Detected custom command, header modified or set by CUSTOM_COMPILE_COMMAND');
    }
    if (!compileArgs.outputFile) {
        logger_1.logger.debug(`pip-compile: implicit output file`);
    }
    // safeguard against index url leak if not explicitly set by an option
    if (compileArgs.commandType === 'pip-compile' &&
        !compileArgs.noEmitIndexUrl &&
        !compileArgs.emitIndexUrl &&
        haveCredentialsInPipEnvironmentVariables()) {
        compileArgs.argv.splice(1, 0, '--no-emit-index-url');
    }
    for (const dep of upgradePackages) {
        compileArgs.argv.push(`--upgrade-package=${(0, shlex_1.quote)(dep.depName + '==' + dep.newVersion)}`);
    }
    return compileArgs.argv.map(shlex_1.quote).join(' ');
}
async function updateArtifacts({ packageFileName: inputFileName, newPackageFileContent: newInputContent, updatedDeps, config, }) {
    if (!config.lockFiles) {
        logger_1.logger.warn({ packageFileName: inputFileName }, 'pip-compile: No lock files associated with a package file');
        return null;
    }
    logger_1.logger.debug(`pipCompile.updateArtifacts(${inputFileName}->${JSON.stringify(config.lockFiles)})`);
    const result = [];
    for (const outputFileName of config.lockFiles) {
        const existingOutput = await (0, fs_1.readLocalFile)(outputFileName, 'utf8');
        if (!existingOutput) {
            logger_1.logger.debug('pip-compile: No output file found');
            return null;
        }
        try {
            await (0, fs_1.writeLocalFile)(inputFileName, newInputContent);
            // TODO(not7cd): use --upgrade option instead deleting
            if (config.isLockFileMaintenance) {
                await (0, fs_1.deleteLocalFile)(outputFileName);
            }
            const compileArgs = (0, common_2.extractHeaderCommand)(existingOutput, outputFileName);
            let pythonVersion;
            if (compileArgs.commandType === 'uv') {
                pythonVersion = compileArgs.pythonVersion;
            }
            else {
                pythonVersion = (0, common_2.extractPythonVersion)(existingOutput, outputFileName);
            }
            const cwd = (0, utils_1.inferCommandExecDir)(outputFileName, compileArgs.outputFile);
            const upgradePackages = updatedDeps.filter((dep) => dep.isLockfileUpdate);
            const packageFiles = [];
            for (const name of compileArgs.sourceFiles) {
                const manager = (0, common_2.matchManager)(name);
                if (manager === 'pip_requirements') {
                    const path = upath_1.default.join(cwd, name);
                    const content = await (0, fs_1.readLocalFile)(path, 'utf8');
                    if (content) {
                        const packageFile = (0, common_1.extractPackageFileFlags)(content);
                        if (packageFile) {
                            packageFiles.push(packageFile);
                        }
                    }
                }
            }
            const cmd = constructPipCompileCmd(compileArgs, upgradePackages);
            const execOptions = await (0, common_2.getExecOptions)(config, compileArgs.commandType, cwd, (0, common_2.getRegistryCredVarsFromPackageFiles)(packageFiles), pythonVersion);
            logger_1.logger.trace({ cwd, cmd }, 'pip-compile command');
            logger_1.logger.trace({ env: execOptions.extraEnv }, 'pip-compile extra env vars');
            await (0, exec_1.exec)(cmd, execOptions);
            const status = await (0, git_1.getRepoStatus)();
            if (status?.modified.includes(outputFileName)) {
                result.push({
                    file: {
                        type: 'addition',
                        path: outputFileName,
                        contents: await (0, fs_1.readLocalFile)(outputFileName, 'utf8'),
                    },
                });
            }
        }
        catch (err) {
            // istanbul ignore if
            if (err.message === error_messages_1.TEMPORARY_ERROR) {
                throw err;
            }
            logger_1.logger.debug({ err }, 'pip-compile: Failed to run command');
            result.push({
                artifactError: {
                    lockFile: outputFileName,
                    stderr: err.message,
                },
            });
        }
    }
    logger_1.logger.debug('pip-compile: Returning updated output file(s)');
    return result.length === 0 ? null : result;
}
//# sourceMappingURL=artifacts.js.map