"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UvLockfileSchema = exports.PdmLockfileSchema = exports.PyProjectSchema = void 0;
const zod_1 = require("zod");
const schema_utils_1 = require("../../../util/schema-utils");
const common_1 = require("../../datasource/pypi/common");
const schema_1 = require("../pixi/schema");
const DependencyListSchema = zod_1.z.array(zod_1.z.string()).optional();
const DependencyRecordSchema = zod_1.z
    .record(zod_1.z.string(), zod_1.z.array(zod_1.z.string()))
    .optional();
const PdmSchema = zod_1.z.object({
    'dev-dependencies': DependencyRecordSchema,
    source: zod_1.z
        .array(zod_1.z.object({
        url: zod_1.z.string(),
        name: zod_1.z.string(),
        verify_ssl: zod_1.z.boolean().optional(),
    }))
        .optional(),
});
const HatchSchema = zod_1.z.object({
    envs: zod_1.z
        .record(zod_1.z.string(), zod_1.z
        .object({
        dependencies: DependencyListSchema,
        'extra-dependencies': DependencyListSchema,
    })
        .optional())
        .optional(),
});
const UvIndexSource = zod_1.z.object({
    index: zod_1.z.string(),
});
const UvGitSource = zod_1.z.object({
    git: zod_1.z.string(),
    rev: zod_1.z.string().optional(),
    tag: zod_1.z.string().optional(),
    branch: zod_1.z.string().optional(),
});
const UvUrlSource = zod_1.z.object({
    url: zod_1.z.string(),
});
const UvPathSource = zod_1.z.object({
    path: zod_1.z.string(),
});
const UvWorkspaceSource = zod_1.z.object({
    workspace: zod_1.z.literal(true),
});
// https://docs.astral.sh/uv/concepts/dependencies/#dependency-sources
const UvSource = zod_1.z.union([
    UvIndexSource,
    UvGitSource,
    UvUrlSource,
    UvPathSource,
    UvWorkspaceSource,
]);
const UvSchema = zod_1.z.object({
    'dev-dependencies': DependencyListSchema,
    'required-version': zod_1.z.string().optional(),
    sources: (0, schema_utils_1.LooseRecord)(
    // uv applies the same normalization as for Python dependencies on sources
    zod_1.z.string().transform((source) => (0, common_1.normalizePythonDepName)(source)), UvSource).optional(),
    index: zod_1.z
        .array(zod_1.z.object({
        name: zod_1.z.string().optional(),
        url: zod_1.z.string(),
        default: zod_1.z.boolean().default(false),
        explicit: zod_1.z.boolean().default(false),
    }))
        .optional(),
});
exports.PyProjectSchema = zod_1.z.object({
    project: zod_1.z
        .object({
        version: zod_1.z.string().optional().catch(undefined),
        'requires-python': zod_1.z.string().optional(),
        dependencies: DependencyListSchema,
        'optional-dependencies': DependencyRecordSchema,
    })
        .optional(),
    'build-system': zod_1.z
        .object({
        requires: DependencyListSchema,
        'build-backend': zod_1.z.string().optional(),
    })
        .optional(),
    'dependency-groups': zod_1.z
        .record(zod_1.z.string(), 
    // Skip non-string entries, like `{include-group = "typing"}`, as they are not dependencies.
    (0, schema_utils_1.LooseArray)(zod_1.z.string()))
        .optional(),
    tool: zod_1.z
        .object({
        pixi: schema_1.PixiConfigSchema.optional(),
        pdm: PdmSchema.optional(),
        hatch: HatchSchema.optional(),
        uv: UvSchema.optional(),
    })
        .optional(),
});
exports.PdmLockfileSchema = schema_utils_1.Toml.pipe(zod_1.z.object({
    package: (0, schema_utils_1.LooseArray)(zod_1.z.object({
        name: zod_1.z.string(),
        version: zod_1.z.string(),
    })),
}))
    .transform(({ package: pkg }) => Object.fromEntries(pkg.map(({ name, version }) => [name, version])))
    .transform((lock) => ({ lock }));
exports.UvLockfileSchema = schema_utils_1.Toml.pipe(zod_1.z.object({
    package: (0, schema_utils_1.LooseArray)(zod_1.z.object({
        name: zod_1.z.string(),
        version: zod_1.z.string(),
    })),
})).transform(({ package: pkg }) => Object.fromEntries(pkg.map(({ name, version }) => [name, version])));
//# sourceMappingURL=schema.js.map