"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createNuGetConfigXml = createNuGetConfigXml;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const hostRules = tslib_1.__importStar(require("../../../util/host-rules"));
const regex_1 = require("../../../util/regex");
const nuget_1 = require("../../datasource/nuget");
const common_1 = require("../../datasource/nuget/common");
function createNuGetConfigXml(registries) {
    let contents = `<?xml version="1.0" encoding="utf-8"?>\n<configuration>\n<packageSources>\n`;
    let unnamedRegistryCount = 0;
    const credentials = [];
    const packageSourceMaps = [];
    for (const registry of registries) {
        const registryName = registry.name ?? `Package source ${++unnamedRegistryCount}`;
        const registryInfo = (0, common_1.parseRegistryUrl)(registry.url);
        contents += formatPackageSourceElement(registryInfo, registryName);
        const { password, username } = hostRules.find({
            hostType: nuget_1.NugetDatasource.id,
            url: registry.url,
        });
        if (is_1.default.nonEmptyString(password) || is_1.default.nonEmptyString(username)) {
            credentials.push({
                name: registryName,
                password,
                username,
            });
        }
        if (registry.sourceMappedPackagePatterns) {
            packageSourceMaps.push({
                name: registryName,
                patterns: registry.sourceMappedPackagePatterns,
            });
        }
    }
    contents += '</packageSources>\n';
    if (credentials.length > 0) {
        contents += '<packageSourceCredentials>\n';
        for (const credential of credentials) {
            contents += formatPackageSourceCredentialElement(credential);
        }
        contents += '</packageSourceCredentials>\n';
    }
    if (packageSourceMaps.length > 0) {
        contents += '<packageSourceMapping>\n';
        for (const packageSourceMap of packageSourceMaps) {
            contents += formatPackageSource(packageSourceMap);
        }
        contents += '</packageSourceMapping>';
    }
    contents += '</configuration>\n';
    return contents;
}
function formatPackageSourceElement(registryInfo, name) {
    let element = `<add key="${name}" value="${registryInfo.feedUrl}" `;
    if (registryInfo.protocolVersion) {
        element += `protocolVersion="${registryInfo.protocolVersion}" `;
    }
    return `${element}/>\n`;
}
function formatPackageSourceCredentialElement(credential) {
    const escapedName = escapeName(credential.name);
    let packageSourceCredential = `<${escapedName}>\n`;
    if (credential.username) {
        packageSourceCredential += `<add key="Username" value="${credential.username}" />\n`;
    }
    if (credential.password) {
        packageSourceCredential += `<add key="ClearTextPassword" value="${credential.password}" />\n`;
    }
    packageSourceCredential += `<add key="ValidAuthenticationTypes" value="basic" />`;
    packageSourceCredential += `</${escapedName}>\n`;
    return packageSourceCredential;
}
function formatPackageSource(packageSourceMap) {
    let packageSource = `<packageSource key="${packageSourceMap.name}">\n`;
    for (const pattern of packageSourceMap.patterns) {
        packageSource += `<package pattern="${pattern}" />\n`;
    }
    return `${packageSource}</packageSource>\n`;
}
const charactersToEscape = (0, regex_1.regEx)(/[^A-Za-z0-9\-_.]/);
function escapeName(name) {
    let escapedName = '';
    for (const char of name) {
        if (char.match(charactersToEscape)) {
            escapedName += `__x${char
                .codePointAt(0)
                .toString(16)
                .padStart(4, '0')}__`;
        }
        else {
            escapedName += char;
        }
    }
    return escapedName;
}
//# sourceMappingURL=config-formatter.js.map