"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseLockFile = parseLockFile;
exports.composeLockFile = composeLockFile;
exports.getNpmrcContent = getNpmrcContent;
exports.updateNpmrcContent = updateNpmrcContent;
exports.resetNpmrcContent = resetNpmrcContent;
const tslib_1 = require("tslib");
const detect_indent_1 = tslib_1.__importDefault(require("detect-indent"));
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
function parseLockFile(lockFile) {
    const detectedIndent = (0, detect_indent_1.default)(lockFile).indent || '  ';
    let lockFileParsed;
    try {
        lockFileParsed = JSON.parse(lockFile);
    }
    catch (err) {
        logger_1.logger.warn({ err }, 'Error parsing npm lock file');
    }
    return { detectedIndent, lockFileParsed };
}
function composeLockFile(lockFile, indent) {
    return JSON.stringify(lockFile, null, indent) + '\n';
}
async function getNpmrcContent(dir) {
    const npmrcFilePath = upath_1.default.join(dir, '.npmrc');
    let originalNpmrcContent = null;
    try {
        originalNpmrcContent = await (0, fs_1.readLocalFile)(npmrcFilePath, 'utf8');
    }
    catch /* istanbul ignore next */ {
        originalNpmrcContent = null;
    }
    if (originalNpmrcContent) {
        logger_1.logger.debug(`npmrc file ${npmrcFilePath} found in repository`);
    }
    return originalNpmrcContent;
}
async function updateNpmrcContent(dir, originalContent, additionalLines) {
    const npmrcFilePath = upath_1.default.join(dir, '.npmrc');
    const newNpmrc = originalContent
        ? [originalContent, ...additionalLines]
        : additionalLines;
    try {
        const newContent = newNpmrc.length ? newNpmrc.join('\n') : null;
        if (newContent !== originalContent) {
            logger_1.logger.debug(`Writing updated .npmrc file to ${npmrcFilePath}`);
            await (0, fs_1.writeLocalFile)(npmrcFilePath, `${newContent}\n`);
        }
    }
    catch /* istanbul ignore next */ {
        logger_1.logger.warn('Unable to write custom npmrc file');
    }
}
async function resetNpmrcContent(dir, originalContent) {
    const npmrcFilePath = upath_1.default.join(dir, '.npmrc');
    if (originalContent) {
        try {
            await (0, fs_1.writeLocalFile)(npmrcFilePath, originalContent);
        }
        catch /* istanbul ignore next */ {
            logger_1.logger.warn('Unable to reset npmrc to original contents');
        }
    }
    else {
        try {
            await (0, fs_1.deleteLocalFile)(npmrcFilePath);
        }
        catch /* istanbul ignore next */ {
            logger_1.logger.warn('Unable to delete custom npmrc');
        }
    }
}
//# sourceMappingURL=utils.js.map