"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.processHostRules = processHostRules;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../../logger");
const hostRules = tslib_1.__importStar(require("../../../../util/host-rules"));
const regex_1 = require("../../../../util/regex");
const string_1 = require("../../../../util/string");
const url_1 = require("../../../../util/url");
function processHostRules() {
    const additionalYarnRcYml = { npmRegistries: {} };
    // Determine the additional npmrc content to add based on host rules
    const additionalNpmrcContent = [];
    const npmHostRules = hostRules.findAll({
        hostType: 'npm',
    });
    logger_1.logger.debug(`Found ${npmHostRules.length} npm host rule(s)`);
    // Include host rules without specific type to mimic the behavior used when determining dependencies with updates.
    const noTypeHostRules = hostRules
        .getAll()
        .filter((rule) => rule.hostType === null || rule.hostType === undefined);
    logger_1.logger.debug(`Found ${noTypeHostRules.length} host rule(s) without host type`);
    // Drop duplicates for the same matchHost while prefering the more specific rules with hostType npm.
    const noTypeHostRulesWithoutDuplicates = noTypeHostRules.filter((rule) => !npmHostRules.some((elem) => elem.matchHost === rule.matchHost));
    logger_1.logger.debug(`Found ${noTypeHostRulesWithoutDuplicates.length} host rule(s) without host type after dropping duplicates`);
    const effectiveHostRules = npmHostRules.concat(noTypeHostRulesWithoutDuplicates);
    logger_1.logger.trace(`Found ${effectiveHostRules.length} effective npm host rule(s) after deduplication`);
    for (const hostRule of effectiveHostRules) {
        if (!hostRule.resolvedHost) {
            logger_1.logger.debug('Skipping host rule without resolved host');
            continue;
        }
        const matchedHost = hostRule.matchHost;
        // Should never be necessary as if we have a resolvedHost, there has to be a matchHost
        // istanbul ignore next
        if (!matchedHost) {
            logger_1.logger.debug('Skipping host rule without matchHost');
            continue;
        }
        const uri = `//${matchedHost}/`;
        let cleanedUri = uri;
        if ((0, url_1.isHttpUrl)(matchedHost)) {
            cleanedUri = matchedHost.replace((0, regex_1.regEx)(/^https?:/), '');
        }
        if (hostRule.token) {
            const key = hostRule.authType === 'Basic' ? '_auth' : '_authToken';
            logger_1.logger.debug(`Adding npmrc entry for ${cleanedUri} with key ${key}`);
            additionalNpmrcContent.push(`${cleanedUri}:${key}=${hostRule.token}`);
            if (hostRule.authType === 'Basic') {
                const registry = {
                    npmAuthIdent: hostRule.token,
                };
                additionalYarnRcYml.npmRegistries[cleanedUri] = registry;
                additionalYarnRcYml.npmRegistries[uri] = registry;
                continue;
            }
            const registry = {
                npmAuthToken: hostRule.token,
            };
            additionalYarnRcYml.npmRegistries[cleanedUri] = registry;
            additionalYarnRcYml.npmRegistries[uri] = registry;
            continue;
        }
        if (is_1.default.string(hostRule.username) && is_1.default.string(hostRule.password)) {
            logger_1.logger.debug(`Adding npmrc entry for ${cleanedUri} with username/password`);
            const password = (0, string_1.toBase64)(hostRule.password);
            additionalNpmrcContent.push(`${cleanedUri}:username=${hostRule.username}`);
            additionalNpmrcContent.push(`${cleanedUri}:_password=${password}`);
            const registries = {
                npmAuthIdent: `${hostRule.username}:${hostRule.password}`,
            };
            additionalYarnRcYml.npmRegistries[cleanedUri] = registries;
            additionalYarnRcYml.npmRegistries[uri] = registries;
        }
    }
    const hasYarnRcNpmRegistries = Object.keys(additionalYarnRcYml.npmRegistries).length > 0;
    return {
        additionalNpmrcContent,
        additionalYarnRcYml: hasYarnRcNpmRegistries
            ? additionalYarnRcYml
            : undefined,
    };
}
//# sourceMappingURL=rules.js.map