"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAquaToolConfig = createAquaToolConfig;
exports.createCargoToolConfig = createCargoToolConfig;
exports.createDotnetToolConfig = createDotnetToolConfig;
exports.createGemToolConfig = createGemToolConfig;
exports.createGoToolConfig = createGoToolConfig;
exports.createNpmToolConfig = createNpmToolConfig;
exports.createPipxToolConfig = createPipxToolConfig;
exports.createSpmToolConfig = createSpmToolConfig;
exports.createUbiToolConfig = createUbiToolConfig;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const regex_1 = require("../../../util/regex");
const crate_1 = require("../../datasource/crate");
const git_refs_1 = require("../../datasource/git-refs");
const git_tags_1 = require("../../datasource/git-tags");
const github_releases_1 = require("../../datasource/github-releases");
const github_tags_1 = require("../../datasource/github-tags");
const go_1 = require("../../datasource/go");
const npm_1 = require("../../datasource/npm");
const nuget_1 = require("../../datasource/nuget");
const pypi_1 = require("../../datasource/pypi");
const common_1 = require("../../datasource/pypi/common");
const rubygems_1 = require("../../datasource/rubygems");
/**
 * Create a tooling config for aqua backend
 * @link https://mise.jdx.dev/dev-tools/backends/aqua.html
 */
function createAquaToolConfig(name, version) {
    // mise supports http aqua package type but we cannot determine it from the tool name
    // An error will be thrown afterwards if the package type is http
    // ref: https://github.com/jdx/mise/blob/d1b9749d8f3e13ef705c1ea471d96c5935b79136/src/aqua/aqua_registry.rs#L39-L45
    return {
        packageName: name,
        datasource: github_tags_1.GithubTagsDatasource.id,
        // Trim the leading 'v' from both the current and extracted version
        currentValue: version.replace((0, regex_1.regEx)(/^v/), ''),
        extractVersion: '^v?(?<version>.+)',
    };
}
const cargoGitVersionRegex = (0, regex_1.regEx)(/^(?<type>tag|branch|rev):(?<version>.+)$/);
/**
 * Create a tooling config for cargo backend
 * @link https://mise.jdx.dev/dev-tools/backends/cargo.html
 */
function createCargoToolConfig(name, version) {
    if (!is_1.default.urlString(name)) {
        return {
            packageName: name, // `is.urlString` type issue
            datasource: crate_1.CrateDatasource.id,
        };
    }
    // tag: branch: or rev: is required for git repository url
    // e.g. branch:main, tag:0.1.0, rev:abcdef
    const matchGroups = cargoGitVersionRegex.exec(version)?.groups;
    if (is_1.default.undefined(matchGroups)) {
        return {
            packageName: name,
            skipReason: 'invalid-version',
        };
    }
    const { type, version: gitVersion } = matchGroups;
    switch (type) {
        case 'tag':
            return {
                packageName: name,
                datasource: git_tags_1.GitTagsDatasource.id,
                currentValue: gitVersion,
            };
        case 'branch':
            return {
                packageName: name,
                datasource: git_refs_1.GitRefsDatasource.id,
                currentValue: gitVersion,
            };
        case 'rev':
            return {
                packageName: name,
                datasource: git_refs_1.GitRefsDatasource.id,
                currentValue: gitVersion,
            };
    }
}
/**
 * Create a tooling config for dotnet backend
 * @link https://mise.jdx.dev/dev-tools/backends/dotnet.html
 */
function createDotnetToolConfig(name) {
    return {
        packageName: name,
        datasource: nuget_1.NugetDatasource.id,
    };
}
/**
 * Create a tooling config for gem backend
 * @link https://mise.jdx.dev/dev-tools/backends/gem.html
 */
function createGemToolConfig(name) {
    return {
        packageName: name,
        datasource: rubygems_1.RubygemsDatasource.id,
    };
}
/**
 * Create a tooling config for go backend
 * @link https://mise.jdx.dev/dev-tools/backends/go.html
 */
function createGoToolConfig(name) {
    return {
        packageName: name,
        datasource: go_1.GoDatasource.id,
    };
}
/**
 * Create a tooling config for npm backend
 * @link https://mise.jdx.dev/dev-tools/backends/npm.html
 */
function createNpmToolConfig(name) {
    return {
        packageName: name,
        datasource: npm_1.NpmDatasource.id,
    };
}
const pipxGitHubRegex = (0, regex_1.regEx)(/^git\+https:\/\/github\.com\/(?<repo>.+)\.git$/);
/**
 * Create a tooling config for pipx backend
 * @link https://mise.jdx.dev/dev-tools/backends/pipx.html
 */
function createPipxToolConfig(name) {
    const isGitSyntax = name.startsWith('git+');
    // Does not support zip file url
    // Avoid type narrowing to prevent type error
    if (!isGitSyntax && is_1.default.urlString(name)) {
        return {
            packageName: name,
            skipReason: 'unsupported-url',
        };
    }
    if (isGitSyntax || name.includes('/')) {
        let repoName;
        if (isGitSyntax) {
            repoName = pipxGitHubRegex.exec(name)?.groups?.repo;
            // If the url is not a github repo, treat the version as a git ref
            if (is_1.default.undefined(repoName)) {
                return {
                    packageName: name.replace(/^git\+/g, '').replaceAll(/\.git$/g, ''),
                    datasource: git_refs_1.GitRefsDatasource.id,
                };
            }
        }
        else {
            repoName = name;
        }
        return {
            packageName: repoName,
            datasource: github_tags_1.GithubTagsDatasource.id,
        };
    }
    return {
        packageName: (0, common_1.normalizePythonDepName)(name),
        datasource: pypi_1.PypiDatasource.id,
    };
}
const spmGitHubRegex = (0, regex_1.regEx)(/^https:\/\/github.com\/(?<repo>.+).git$/);
/**
 * Create a tooling config for spm backend
 * @link https://mise.jdx.dev/dev-tools/backends/spm.html
 */
function createSpmToolConfig(name) {
    let repoName;
    // Avoid type narrowing to prevent type error
    if (is_1.default.urlString(name)) {
        repoName = spmGitHubRegex.exec(name)?.groups?.repo;
        // spm backend only supports github repos
        if (!repoName) {
            return {
                packageName: name,
                skipReason: 'unsupported-url',
            };
        }
    }
    return {
        packageName: repoName ?? name,
        datasource: github_releases_1.GithubReleasesDatasource.id,
    };
}
/**
 * Create a tooling config for ubi backend
 * @link https://mise.jdx.dev/dev-tools/backends/ubi.html
 */
function createUbiToolConfig(name, version, toolOptions) {
    let extractVersion = undefined;
    const hasVPrefix = version.startsWith('v');
    const setsTagRegex = !hasVPrefix || is_1.default.string(toolOptions.tag_regex);
    if (setsTagRegex) {
        // By default, use a regex that matches any tag
        let tagRegex = '.+';
        // Filter versions by tag_regex if it is specified
        // ref: https://mise.jdx.dev/dev-tools/backends/ubi.html#ubi-uses-weird-versions
        if (is_1.default.string(toolOptions.tag_regex)) {
            // Remove the leading '^' if it exists to avoid duplication
            tagRegex = toolOptions.tag_regex.replace(/^\^/, '');
            if (!hasVPrefix) {
                // Remove the leading 'v?' if it exists to avoid duplication
                tagRegex = tagRegex.replace(/^v\??/, '');
            }
        }
        // Trim the 'v' prefix if the current version does not have it
        extractVersion = `^${hasVPrefix ? '' : 'v?'}(?<version>${tagRegex})`;
    }
    return {
        packageName: name,
        datasource: github_releases_1.GithubReleasesDatasource.id,
        currentValue: version,
        ...(is_1.default.string(extractVersion) ? { extractVersion } : {}),
    };
}
//# sourceMappingURL=backends.js.map