"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const shlex_1 = require("shlex");
const upath_1 = tslib_1.__importDefault(require("upath"));
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const git_1 = require("../../../util/git");
const docker_1 = require("../../datasource/docker");
const helm_1 = require("../../datasource/helm");
const common_1 = require("./common");
const extract_1 = require("./extract");
async function localExistingChartPath(chartHome, dependencyName, version) {
    const folderName = `${dependencyName}-${version}`;
    const path = upath_1.default.join(chartHome, folderName);
    const pathExists = await (0, fs_1.localPathExists)(path);
    return pathExists ? path : null;
}
function helmRepositoryArgs(repository, depName, datasource) {
    switch (datasource) {
        case helm_1.HelmDatasource.id:
            return `--repo ${(0, shlex_1.quote)(repository)} ${(0, shlex_1.quote)(depName)}`;
        case docker_1.DockerDatasource.id:
            return (0, shlex_1.quote)(`oci://${repository}`);
        /* v8 ignore next 2: should never happen */
        default:
            throw new Error(`Unknown datasource: ${datasource}`);
    }
}
async function inflateHelmChart(flagEnabled, execOptions, chartHome, depName, repository, currentVersion, newVersion, datasource) {
    const currentChartExistingPath = await localExistingChartPath(chartHome, depName, currentVersion);
    if (!flagEnabled && is_1.default.nullOrUndefined(currentChartExistingPath)) {
        logger_1.logger.debug(`Not inflating Helm chart for ${depName} as kustomizeInflateHelmCharts is not enabled and the current version isn't inflated`);
        return;
    }
    if (is_1.default.nonEmptyString(currentChartExistingPath) &&
        is_1.default.nonEmptyString(newVersion)) {
        logger_1.logger.debug(`Deleting previous helm chart: ${currentChartExistingPath}`);
        await (0, fs_1.deleteLocalFile)(currentChartExistingPath);
    }
    const versionToPull = newVersion ?? currentVersion;
    const versionToPullExistingPath = await localExistingChartPath(chartHome, depName, versionToPull);
    if (is_1.default.nonEmptyString(versionToPullExistingPath)) {
        logger_1.logger.debug(`Helm chart ${depName} version ${versionToPull} already exists at ${versionToPullExistingPath}`);
        return;
    }
    const folderName = `${depName}-${versionToPull}`;
    const untarDir = upath_1.default.join(chartHome, folderName);
    logger_1.logger.debug(`Pulling helm chart ${depName} version ${versionToPull} to ${untarDir}`);
    const cmd = `helm pull --untar --untardir ${(0, shlex_1.quote)(untarDir)} ` +
        `--version ${(0, shlex_1.quote)(versionToPull)} ${helmRepositoryArgs(repository, depName, datasource)}`;
    await (0, exec_1.exec)(cmd, execOptions);
}
async function updateArtifacts({ packageFileName, updatedDeps, newPackageFileContent, config, }) {
    logger_1.logger.debug(`kustomize.updateArtifacts(${packageFileName})`);
    const project = (0, extract_1.parseKustomize)(newPackageFileContent);
    const isUpdateOptionInflateChartArchives = config.postUpdateOptions?.includes('kustomizeInflateHelmCharts') === true;
    if (is_1.default.nullOrUndefined(project)) {
        return [
            {
                artifactError: {
                    stderr: 'Failed to parse new package file content',
                },
            },
        ];
    }
    const chartHome = (0, fs_1.getSiblingFileName)(packageFileName, project.helmGlobals?.chartHome ?? 'charts');
    try {
        const helmToolConstraint = {
            toolName: 'helm',
            constraint: config.constraints?.helm,
        };
        const execOptions = {
            docker: {},
            extraEnv: (0, common_1.generateHelmEnvs)(config),
            toolConstraints: [helmToolConstraint],
        };
        for (const dependency of updatedDeps) {
            if (!dependency.currentVersion) {
                continue;
            }
            if (dependency.newVersion === dependency.currentVersion) {
                continue;
            }
            if (!is_1.default.nonEmptyString(dependency.depName)) {
                continue;
            }
            if (dependency.depType !== 'HelmChart') {
                continue;
            }
            let repository = null;
            switch (dependency.datasource) {
                case helm_1.HelmDatasource.id:
                    repository = dependency.registryUrls?.[0];
                    break;
                case docker_1.DockerDatasource.id:
                    repository = dependency.packageName;
                    break;
            }
            if (is_1.default.nullOrUndefined(repository)) {
                continue;
            }
            await inflateHelmChart(isUpdateOptionInflateChartArchives, execOptions, chartHome, dependency.depName, repository, dependency.currentVersion, dependency.newVersion, dependency.datasource);
        }
        const status = await (0, git_1.getRepoStatus)();
        const chartsAddition = status?.not_added ?? [];
        const chartsDeletion = status?.deleted ?? [];
        const fileChanges = [];
        for (const file of chartsAddition) {
            // only add artifacts in the chartHome path
            if (!file.startsWith(chartHome)) {
                continue;
            }
            fileChanges.push({
                file: {
                    type: 'addition',
                    path: file,
                    contents: await (0, fs_1.readLocalFile)(file),
                },
            });
        }
        for (const file of chartsDeletion) {
            // only add artifacts in the chartHome path
            if (!file.startsWith(chartHome)) {
                continue;
            }
            fileChanges.push({
                file: {
                    type: 'deletion',
                    path: file,
                },
            });
        }
        return fileChanges.length > 0 ? fileChanges : null;
    }
    catch (err) {
        if (err.message === error_messages_1.TEMPORARY_ERROR) {
            throw err;
        }
        logger_1.logger.debug({ err }, 'Failed to inflate helm chart');
        return [
            {
                artifactError: {
                    stderr: err.message,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map