"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const yaml_1 = require("../../../util/yaml");
const helm_1 = require("../../datasource/helm");
const utils_1 = require("./utils");
async function extractPackageFile(content, packageFile, config) {
    let chart;
    try {
        // TODO: use schema (#9610)
        chart = (0, yaml_1.parseSingleYaml)(content);
        if (!(chart?.apiVersion && chart.name && chart.version)) {
            logger_1.logger.debug({ packageFile }, 'Failed to find required fields in Chart.yaml');
            return null;
        }
        if (chart.apiVersion !== 'v2') {
            logger_1.logger.debug({ packageFile }, 'Unsupported Chart apiVersion. Only v2 is supported.');
            return null;
        }
    }
    catch {
        logger_1.logger.debug({ packageFile }, `Failed to parse helm Chart.yaml`);
        return null;
    }
    const packageFileVersion = chart.version;
    let deps = [];
    if (!is_1.default.nonEmptyArray(chart?.dependencies)) {
        logger_1.logger.debug(`Chart has no dependencies in ${packageFile}`);
        return null;
    }
    const validDependencies = chart.dependencies.filter((dep) => is_1.default.nonEmptyString(dep.name) && is_1.default.nonEmptyString(dep.version));
    if (!is_1.default.nonEmptyArray(validDependencies)) {
        logger_1.logger.debug('Name and/or version missing for all dependencies');
        return null;
    }
    deps = validDependencies.map((dep) => {
        const res = {
            depName: dep.name,
            currentValue: dep.version,
        };
        if (!dep.repository) {
            res.skipReason = 'no-repository';
            return res;
        }
        const repository = (0, utils_1.resolveAlias)(dep.repository, config.registryAliases);
        if (!repository) {
            res.skipReason = 'placeholder-url';
            return res;
        }
        const result = {
            ...res,
            ...(0, utils_1.parseRepository)(dep.name, repository),
        };
        return result;
    });
    const res = {
        deps,
        datasource: helm_1.HelmDatasource.id,
        packageFileVersion,
    };
    const lockFileName = (0, fs_1.getSiblingFileName)(packageFile, 'Chart.lock');
    // istanbul ignore if
    if (await (0, fs_1.localPathExists)(lockFileName)) {
        res.lockFiles = [lockFileName];
    }
    return res;
}
//# sourceMappingURL=extract.js.map