"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateLoginCmd = generateLoginCmd;
exports.generateHelmEnvs = generateHelmEnvs;
const tslib_1 = require("tslib");
const shlex_1 = require("shlex");
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const fs_1 = require("../../../util/fs");
const sanitize_1 = require("../../../util/sanitize");
const string_1 = require("../../../util/string");
const ecr_1 = require("../../datasource/docker/ecr");
async function generateLoginCmd(repositoryRule) {
    logger_1.logger.trace({ repositoryRule }, 'Generating Helm registry login command');
    const { hostRule, repository } = repositoryRule;
    const { username, password } = hostRule;
    const loginCMD = 'helm registry login';
    if (username !== 'AWS' && ecr_1.ecrRegex.test(repository)) {
        logger_1.logger.trace({ repository }, `Using ecr auth for Helm registry`);
        const [, region] = (0, array_1.coerceArray)(ecr_1.ecrRegex.exec(repository));
        const auth = await (0, ecr_1.getECRAuthToken)(region, hostRule);
        if (!auth) {
            return null;
        }
        const [username, password] = (0, string_1.fromBase64)(auth).split(':');
        if (!username || !password) {
            return null;
        }
        (0, sanitize_1.addSecretForSanitizing)(username);
        (0, sanitize_1.addSecretForSanitizing)(password);
        return `${loginCMD} --username ${(0, shlex_1.quote)(username)} --password ${(0, shlex_1.quote)(password)} ${(0, shlex_1.quote)(repository)}`;
    }
    if (username && password) {
        logger_1.logger.trace({ repository }, `Using basic auth for Helm registry`);
        // Split off any path as it's not valid for the helm registry login command
        const hostPart = repository.split('/')[0];
        const cmd = `${loginCMD} --username ${(0, shlex_1.quote)(username)} --password ${(0, shlex_1.quote)(password)} ${(0, shlex_1.quote)(hostPart)}`;
        logger_1.logger.trace({ cmd }, 'Generated Helm registry login command');
        return cmd;
    }
    return null;
}
function generateHelmEnvs() {
    return {
        HELM_EXPERIMENTAL_OCI: '1',
        // set cache and config files to a path in privateCacheDir to prevent file and credential leakage
        HELM_REGISTRY_CONFIG: `${upath_1.default.join((0, fs_1.privateCacheDir)(), 'registry.json')}`,
        HELM_REPOSITORY_CONFIG: `${upath_1.default.join((0, fs_1.privateCacheDir)(), 'repositories.yaml')}`,
        HELM_REPOSITORY_CACHE: `${upath_1.default.join((0, fs_1.privateCacheDir)(), 'repositories')}`,
    };
}
//# sourceMappingURL=common.js.map