"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseGradle = parseGradle;
exports.parseKotlinSource = parseKotlinSource;
exports.parseJavaToolchainVersion = parseJavaToolchainVersion;
exports.parseProps = parseProps;
const good_enough_parser_1 = require("good-enough-parser");
const regex_1 = require("../../../util/regex");
const apply_from_1 = require("./parser/apply-from");
const assignments_1 = require("./parser/assignments");
const common_1 = require("./parser/common");
const dependencies_1 = require("./parser/dependencies");
const handlers_1 = require("./parser/handlers");
const language_version_1 = require("./parser/language-version");
const objects_1 = require("./parser/objects");
const plugins_1 = require("./parser/plugins");
const registry_urls_1 = require("./parser/registry-urls");
const version_catalogs_1 = require("./parser/version-catalogs");
const utils_1 = require("./utils");
const groovy = good_enough_parser_1.lang.createLang('groovy');
const ctx = {
    packageFile: '',
    fileContents: {},
    recursionDepth: 0,
    globalVars: {},
    deps: [],
    registryUrls: [],
    varTokens: [],
    tmpKotlinImportStore: [],
    tmpNestingDepth: [],
    tmpRegistryContent: [],
    tmpTokenStore: {},
    tokenMap: {},
};
(0, handlers_1.setParseGradleFunc)(parseGradle);
function parseGradle(input, initVars = {}, packageFile = '', fileContents = {}, recursionDepth = 0) {
    let vars = { ...initVars };
    const deps = [];
    const urls = [];
    const query = good_enough_parser_1.query.tree({
        type: 'root-tree',
        search: good_enough_parser_1.query.alt(common_1.qKotlinImport, assignments_1.qAssignments, dependencies_1.qDependencies, plugins_1.qPlugins, registry_urls_1.qRegistryUrls, version_catalogs_1.qVersionCatalogs, dependencies_1.qLongFormDep, apply_from_1.qApplyFrom),
    });
    const parsedResult = groovy.query(input, query, {
        ...ctx,
        packageFile,
        fileContents,
        recursionDepth,
        globalVars: vars,
    });
    if (parsedResult) {
        deps.push(...parsedResult.deps);
        vars = { ...vars, ...parsedResult.globalVars };
        urls.push(...parsedResult.registryUrls);
    }
    return { deps, urls, vars };
}
function parseKotlinSource(input, initVars = {}, packageFile = '') {
    let vars = { ...initVars };
    const deps = [];
    const query = good_enough_parser_1.query.tree({
        type: 'root-tree',
        maxDepth: 1,
        search: objects_1.qKotlinMultiObjectVarAssignment,
    });
    const parsedResult = groovy.query(input, query, {
        ...ctx,
        packageFile,
        globalVars: vars,
    });
    if (parsedResult) {
        deps.push(...parsedResult.deps);
        vars = { ...vars, ...parsedResult.globalVars };
    }
    return { deps, vars };
}
function parseJavaToolchainVersion(input) {
    const ctx = {};
    const parsedResult = groovy.query(input, language_version_1.qToolchainVersion, ctx);
    return parsedResult?.javaLanguageVersion ?? null;
}
const propWord = '[a-zA-Z_][a-zA-Z0-9_]*(?:\\.[a-zA-Z_][a-zA-Z0-9_]*)*';
const propRegex = (0, regex_1.regEx)(`^(?<leftPart>\\s*(?<key>${propWord})\\s*[= :]\\s*['"]?)(?<value>[^\\s'"]+)['"]?\\s*$`);
function parseProps(input, packageFile) {
    let offset = 0;
    const vars = {};
    const deps = [];
    for (const line of input.split(regex_1.newlineRegex)) {
        const lineMatch = propRegex.exec(line);
        if (lineMatch?.groups) {
            const { key, value, leftPart } = lineMatch.groups;
            const replacePosition = offset + leftPart.length;
            const dep = (0, utils_1.parseDependencyString)(value);
            if (dep) {
                deps.push({
                    ...dep,
                    managerData: {
                        fileReplacePosition: replacePosition + dep.depName.length + 1,
                        packageFile,
                    },
                });
            }
            else {
                vars[key] = {
                    key,
                    value,
                    fileReplacePosition: replacePosition,
                    packageFile,
                };
            }
        }
        offset += line.length + 1;
    }
    return { vars, deps };
}
//# sourceMappingURL=parser.js.map