"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.matchesContentDescriptor = matchesContentDescriptor;
exports.extractAllPackageFiles = extractAllPackageFiles;
const tslib_1 = require("tslib");
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const fs_1 = require("../../../util/fs");
const regex_1 = require("../../../util/regex");
const maven_1 = require("../../datasource/maven");
const gradle_1 = tslib_1.__importDefault(require("../../versioning/gradle"));
const catalog_1 = require("./extract/catalog");
const consistent_versions_plugin_1 = require("./extract/consistent-versions-plugin");
const parser_1 = require("./parser");
const common_1 = require("./parser/common");
const utils_1 = require("./utils");
const mavenDatasource = maven_1.MavenDatasource.id;
function updatePackageRegistries(packageRegistries, urls) {
    for (const url of urls) {
        const registryAlreadyKnown = packageRegistries.some((item) => item.registryUrl === url.registryUrl &&
            item.scope === url.scope &&
            item.registryType === url.registryType &&
            item.content === url.content);
        if (!registryAlreadyKnown) {
            packageRegistries.push(url);
        }
    }
}
function matchesContentDescriptor(dep, contentDescriptors) {
    const [groupId, artifactId] = (dep.packageName ?? dep.depName).split(':');
    let hasIncludes = false;
    let hasExcludes = false;
    let matchesInclude = false;
    let matchesExclude = false;
    for (const content of (0, array_1.coerceArray)(contentDescriptors)) {
        const { mode, matcher, groupId: contentGroupId, artifactId: contentArtifactId, version: contentVersion, } = content;
        // group matching
        let groupMatch = false;
        if (matcher === 'regex') {
            groupMatch = (0, regex_1.regEx)(contentGroupId).test(groupId);
        }
        else if (matcher === 'subgroup') {
            groupMatch =
                groupId === contentGroupId || `${groupId}.`.startsWith(contentGroupId);
        }
        else {
            groupMatch = groupId === contentGroupId;
        }
        // artifact matching (optional)
        let artifactMatch = true;
        if (groupMatch && contentArtifactId) {
            if (matcher === 'regex') {
                artifactMatch = (0, regex_1.regEx)(contentArtifactId).test(artifactId);
            }
            else {
                artifactMatch = artifactId === contentArtifactId;
            }
        }
        // version matching (optional)
        let versionMatch = true;
        if (groupMatch && artifactMatch && contentVersion && dep.currentValue) {
            if (matcher === 'regex') {
                versionMatch = (0, regex_1.regEx)(contentVersion).test(dep.currentValue);
            }
            else {
                // contentVersion can be an exact version or a gradle-supported version range
                versionMatch = gradle_1.default.matches(dep.currentValue, contentVersion);
            }
        }
        const isMatch = groupMatch && artifactMatch && versionMatch;
        if (mode === 'include') {
            hasIncludes = true;
            if (isMatch) {
                matchesInclude = true;
            }
        }
        else if (mode === 'exclude') {
            hasExcludes = true;
            if (isMatch) {
                matchesExclude = true;
            }
        }
    }
    if (hasIncludes && hasExcludes) {
        // if both includes and excludes exist, dep must match include and not match exclude
        return matchesInclude && !matchesExclude;
    }
    else if (hasIncludes) {
        // if only includes exist, dep must match at least one include
        return matchesInclude;
    }
    else if (hasExcludes) {
        // if only excludes exist, dep must not match any exclude
        return !matchesExclude;
    }
    // by default, repositories include everything and exclude nothing
    return true;
}
function getRegistryUrlsForDep(packageRegistries, dep) {
    const scope = dep.depType === 'plugin' ? 'plugin' : 'dep';
    const matchingRegistries = packageRegistries.filter((item) => item.scope === scope && matchesContentDescriptor(dep, item.content));
    const exclusiveRegistries = matchingRegistries.filter((item) => item.registryType === 'exclusive');
    const registryUrls = (exclusiveRegistries.length ? exclusiveRegistries : matchingRegistries).map((item) => item.registryUrl);
    if (!registryUrls.length && scope === 'plugin') {
        registryUrls.push(common_1.REGISTRY_URLS.gradlePluginPortal);
    }
    return [...new Set(registryUrls)];
}
async function parsePackageFiles(config, packageFiles, extractedDeps, packageFilesByName, packageRegistries) {
    const varRegistry = {};
    const fileContents = await (0, fs_1.getLocalFiles)(packageFiles);
    for (const packageFile of packageFiles) {
        packageFilesByName[packageFile] = {
            packageFile,
            datasource: mavenDatasource,
            deps: [],
        };
        try {
            // TODO #22198
            const content = fileContents[packageFile];
            const packageFileDir = upath_1.default.dirname((0, utils_1.toAbsolutePath)(packageFile));
            if ((0, utils_1.isPropsFile)(packageFile)) {
                const { vars, deps } = (0, parser_1.parseProps)(content, packageFile);
                (0, utils_1.updateVars)(varRegistry, packageFileDir, vars);
                extractedDeps.push(...deps);
            }
            else if ((0, utils_1.isTOMLFile)(packageFile)) {
                const deps = (0, catalog_1.parseCatalog)(packageFile, content);
                extractedDeps.push(...deps);
            }
            else if ((0, consistent_versions_plugin_1.isGcvPropsFile)(packageFile) &&
                (0, consistent_versions_plugin_1.usesGcv)(packageFile, fileContents)) {
                const deps = (0, consistent_versions_plugin_1.parseGcv)(packageFile, fileContents);
                extractedDeps.push(...deps);
            }
            else if ((0, utils_1.isKotlinSourceFile)(packageFile)) {
                const vars = (0, utils_1.getVars)(varRegistry, packageFileDir);
                const { vars: gradleVars, deps } = (0, parser_1.parseKotlinSource)(content, vars, packageFile);
                (0, utils_1.updateVars)(varRegistry, '/', gradleVars);
                extractedDeps.push(...deps);
            }
            else if ((0, utils_1.isGradleScriptFile)(packageFile)) {
                const vars = (0, utils_1.getVars)(varRegistry, packageFileDir);
                const { deps, urls, vars: gradleVars, } = (0, parser_1.parseGradle)(content, vars, packageFile, fileContents);
                updatePackageRegistries(packageRegistries, urls);
                (0, utils_1.updateVars)(varRegistry, packageFileDir, gradleVars);
                extractedDeps.push(...deps);
            }
        }
        catch (err) {
            logger_1.logger.debug({ err, config, packageFile }, `Failed to process Gradle file`);
        }
    }
    return extractedDeps;
}
async function extractAllPackageFiles(config, packageFiles) {
    const packageFilesByName = {};
    const packageRegistries = [];
    const extractedDeps = [];
    const kotlinSourceFiles = packageFiles.filter(utils_1.isKotlinSourceFile);
    const gradleFiles = (0, utils_1.reorderFiles)(packageFiles.filter((e) => !kotlinSourceFiles.includes(e)));
    await parsePackageFiles(config, [...kotlinSourceFiles, ...kotlinSourceFiles, ...gradleFiles], extractedDeps, packageFilesByName, packageRegistries);
    if (!extractedDeps.length) {
        return null;
    }
    for (const dep of extractedDeps) {
        dep.fileReplacePosition = dep?.managerData?.fileReplacePosition; // #8224
        const key = dep.managerData?.packageFile;
        // istanbul ignore else
        if (key) {
            let pkgFile = packageFilesByName[key];
            // istanbul ignore if: won't happen if "apply from" processes only initially known files
            if (!pkgFile) {
                pkgFile = {
                    packageFile: key,
                    datasource: mavenDatasource,
                    deps: [],
                };
            }
            dep.datasource ??= mavenDatasource;
            if (dep.datasource === mavenDatasource) {
                dep.registryUrls = getRegistryUrlsForDep(packageRegistries, dep);
                dep.depType ??=
                    key.startsWith('buildSrc') && !kotlinSourceFiles.length
                        ? 'devDependencies'
                        : 'dependencies';
            }
            const depAlreadyInPkgFile = pkgFile.deps.some((item) => item.depName === dep.depName &&
                item.managerData?.fileReplacePosition ===
                    dep.managerData?.fileReplacePosition);
            if (!depAlreadyInPkgFile) {
                pkgFile.deps.push(dep);
            }
            packageFilesByName[key] = pkgFile;
        }
        else {
            logger_1.logger.debug({ dep }, `Failed to process Gradle dependency`);
        }
    }
    return Object.values(packageFilesByName);
}
//# sourceMappingURL=extract.js.map