"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
exports.extractAllPackageFiles = extractAllPackageFiles;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const fs_1 = require("../../../util/fs");
const yaml_1 = require("../../../util/yaml");
const glasskube_packages_1 = require("../../datasource/glasskube-packages");
const schema_1 = require("./schema");
function parseResources(content, packageFile) {
    const resources = (0, yaml_1.parseYaml)(content, {
        customSchema: schema_1.GlasskubeResource,
        failureBehaviour: 'filter',
    });
    const packages = [];
    const repositories = [];
    for (const resource of resources) {
        if (resource.kind === 'ClusterPackage' || resource.kind === 'Package') {
            packages.push(resource);
        }
        else if (resource.kind === 'PackageRepository') {
            repositories.push(resource);
        }
    }
    return { packageFile, repositories, packages };
}
function resolvePackageDependencies(packages, repositories) {
    const deps = [];
    for (const pkg of packages) {
        const dep = {
            depName: pkg.spec.packageInfo.name,
            currentValue: pkg.spec.packageInfo.version,
            datasource: glasskube_packages_1.GlasskubePackagesDatasource.id,
        };
        const repository = findRepository(pkg.spec.packageInfo.repositoryName ?? null, repositories);
        if (repository === null) {
            dep.skipReason = 'unknown-registry';
        }
        else {
            dep.registryUrls = [repository.spec.url];
        }
        deps.push(dep);
    }
    return deps;
}
function findRepository(name, repositories) {
    for (const repository of repositories) {
        if (name === repository.metadata.name) {
            return repository;
        }
        if (is_1.default.falsy(name) && isDefaultRepository(repository)) {
            return repository;
        }
    }
    return null;
}
function isDefaultRepository(repository) {
    return (repository.metadata.annotations?.['packages.glasskube.dev/default-repository'] === 'true');
}
function extractPackageFile(content, packageFile, config) {
    const { packages, repositories } = parseResources(content, packageFile);
    const deps = resolvePackageDependencies(packages, repositories);
    return { deps };
}
async function extractAllPackageFiles(config, packageFiles) {
    const allRepositories = [];
    const glasskubeResourceFiles = [];
    for (const packageFile of packageFiles) {
        const content = await (0, fs_1.readLocalFile)(packageFile, 'utf8');
        if (content !== null) {
            const resources = parseResources(content, packageFile);
            allRepositories.push(...resources.repositories);
            glasskubeResourceFiles.push(resources);
        }
    }
    const result = [];
    for (const file of glasskubeResourceFiles) {
        const deps = resolvePackageDependencies(file.packages, allRepositories);
        if (deps.length > 0) {
            result.push({ packageFile: file.packageFile, deps });
        }
    }
    return result.length ? result : null;
}
//# sourceMappingURL=extract.js.map