"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
exports.extractAllPackageFiles = extractAllPackageFiles;
const logger_1 = require("../../../logger");
const fs_1 = require("../../../util/fs");
const regex_1 = require("../../../util/regex");
const result_1 = require("../../../util/result");
const yaml_1 = require("../../../util/yaml");
const gitlab_tags_1 = require("../../datasource/gitlab-tags");
const schema_1 = require("./schema");
const utils_1 = require("./utils");
// See https://docs.gitlab.com/ee/ci/components/index.html#use-a-component
const componentReferenceRegex = (0, regex_1.regEx)(/(?<fqdn>[^/]+)\/(?<projectPath>.+)\/(?:.+)@(?<specificVersion>.+)/);
const componentReferenceLatestVersion = '~latest';
function extractDepFromIncludeComponent(component, registryAliases) {
    let componentUrl = component;
    if (registryAliases) {
        for (const key in registryAliases) {
            componentUrl = componentUrl.replace(key, registryAliases[key]);
        }
    }
    const componentReference = componentReferenceRegex.exec(componentUrl)?.groups;
    if (!componentReference) {
        logger_1.logger.debug({ componentReference: componentUrl }, 'Ignoring malformed component reference');
        return null;
    }
    const projectPathParts = componentReference.projectPath.split('/');
    if (projectPathParts.length < 2) {
        logger_1.logger.debug({ componentReference: componentUrl }, 'Ignoring component reference with incomplete project path');
        return null;
    }
    const dep = {
        datasource: gitlab_tags_1.GitlabTagsDatasource.id,
        depName: componentReference.projectPath,
        depType: 'repository',
        currentValue: componentReference.specificVersion,
        registryUrls: [`https://${componentReference.fqdn}`],
    };
    if (dep.currentValue === componentReferenceLatestVersion) {
        logger_1.logger.debug({ componentVersion: dep.currentValue }, 'Ignoring component version');
        dep.skipReason = 'unsupported-version';
    }
    return dep;
}
function extractPackageFile(content, packageFile, config) {
    const deps = [];
    try {
        const docs = (0, yaml_1.parseYaml)(content, { uniqueKeys: false });
        for (const doc of docs) {
            const topLevel = schema_1.Job.parse(doc);
            const jobs = schema_1.Jobs.parse(doc);
            for (const job of [topLevel, ...jobs]) {
                const { image, services } = job;
                if (image) {
                    const dep = (0, utils_1.getGitlabDep)(image.value, config.registryAliases);
                    dep.depType = image.type;
                    deps.push(dep);
                }
                for (const service of services) {
                    const dep = (0, utils_1.getGitlabDep)(service, config.registryAliases);
                    dep.depType = 'service-image';
                    deps.push(dep);
                }
            }
            const includedComponents = schema_1.GitlabDocument.parse(doc);
            for (const includedComponent of includedComponents) {
                const dep = extractDepFromIncludeComponent(includedComponent, config.registryAliases);
                if (dep) {
                    deps.push(dep);
                }
            }
        }
    }
    catch (err) /* istanbul ignore next */ {
        if (err.stack?.startsWith('YAMLException:')) {
            logger_1.logger.debug({ err, packageFile }, 'YAML exception extracting GitLab CI includes');
        }
        else {
            logger_1.logger.debug({ err, packageFile }, 'Error extracting GitLab CI dependencies');
        }
    }
    return deps.length ? { deps } : null;
}
async function extractAllPackageFiles(config, packageFiles) {
    const filesToExamine = [...packageFiles];
    const seen = new Set(packageFiles);
    const results = [];
    // extract all includes from the files
    while (filesToExamine.length > 0) {
        const file = filesToExamine.pop();
        const content = await (0, fs_1.readLocalFile)(file, 'utf8');
        if (!content) {
            logger_1.logger.debug({ packageFile: file }, `Empty or non existent gitlabci file`);
            continue;
        }
        const { val: docs, err } = result_1.Result.wrap(() => (0, yaml_1.parseYaml)(content, { uniqueKeys: false })).unwrap();
        if (err) {
            logger_1.logger.debug({ err, packageFile: file }, 'Error extracting GitLab CI dependencies');
            continue;
        }
        const localIncludes = schema_1.MultiDocumentLocalIncludes.parse(docs);
        for (const file of localIncludes) {
            if (!seen.has(file)) {
                seen.add(file);
                filesToExamine.push(file);
            }
        }
        const result = extractPackageFile(content, file, config);
        if (result !== null) {
            results.push({
                packageFile: file,
                deps: result.deps,
            });
        }
    }
    logger_1.logger.trace({ packageFiles, files: filesToExamine.entries() }, 'extracted all GitLab CI files');
    if (!results.length) {
        return null;
    }
    return results;
}
//# sourceMappingURL=extract.js.map