"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractPackageFile = extractPackageFile;
const logger_1 = require("../../../logger");
const golang_version_1 = require("../../datasource/golang-version");
const node_version_1 = require("../../datasource/node-version");
const python_version_1 = require("../../datasource/python-version");
const ruby_version_1 = require("../../datasource/ruby-version");
const utils_1 = require("../custom/utils");
const extract_1 = require("../dockerfile/extract");
const schema_1 = require("./schema");
function extractPackageFile(content, packageFile, extractConfig) {
    try {
        const file = schema_1.DevContainerFile.parse(content);
        const deps = [];
        const image = file?.image ?? null;
        const imageDep = getDep(image, packageFile, extractConfig.registryAliases);
        if (imageDep) {
            imageDep.depType = 'image';
            deps.push(imageDep);
        }
        else {
            logger_1.logger.trace({ packageFile }, 'No image defined in dev container JSON file.');
        }
        const features = file.features;
        if (features) {
            for (const [feature, value] of Object.entries(features)) {
                const featureDep = getDep(feature, packageFile, extractConfig.registryAliases);
                if (featureDep) {
                    featureDep.depType = 'feature';
                    featureDep.pinDigests = false;
                    deps.push(featureDep);
                    let dep;
                    switch (featureDep.depName) {
                        case 'ghcr.io/devcontainers/features/node':
                            dep = {
                                depName: 'node',
                                datasource: node_version_1.NodeVersionDatasource.id,
                                currentValue: value.version,
                            };
                            break;
                        case 'ghcr.io/devcontainers/features/go':
                            dep = {
                                depName: 'go',
                                datasource: golang_version_1.GolangVersionDatasource.id,
                                currentValue: value.version,
                            };
                            break;
                        case 'ghcr.io/devcontainers/features/python':
                            dep = {
                                depName: 'python',
                                datasource: python_version_1.PythonVersionDatasource.id,
                                currentValue: value.version,
                            };
                            break;
                        case 'ghcr.io/devcontainers/features/ruby':
                            dep = {
                                depName: 'ruby',
                                datasource: ruby_version_1.RubyVersionDatasource.id,
                                currentValue: value.version,
                            };
                            break;
                        default:
                            // skip additional checks if not a known feature
                            continue;
                    }
                    if (!value.version) {
                        dep.skipReason = 'unspecified-version';
                    }
                    deps.push(dep);
                    continue;
                }
                logger_1.logger.trace({ feature, packageFile }, 'Skipping invalid dependency in dev container JSON file.');
            }
        }
        if (deps.length < 1) {
            logger_1.logger.trace({ packageFile }, 'No dependencies to process for dev container JSON file.');
            return null;
        }
        return { deps };
    }
    catch (err) {
        logger_1.logger.debug({ err, packageFile }, 'Error extracting dev container JSON file.');
        return null;
    }
}
function getDep(subject, packageFile, registryAliases) {
    if (!subject) {
        return null;
    }
    const dep = (0, extract_1.getDep)(subject, true, registryAliases);
    if (!(0, utils_1.isValidDependency)(dep)) {
        logger_1.logger.trace({ subject, packageFile }, 'Skipping invalid docker dependency in dev container JSON file.');
        return null;
    }
    return dep;
}
//# sourceMappingURL=extract.js.map