"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateArtifacts = updateArtifacts;
const shlex_1 = require("shlex");
const error_messages_1 = require("../../../constants/error-messages");
const logger_1 = require("../../../logger");
const exec_1 = require("../../../util/exec");
const fs_1 = require("../../../util/fs");
const auth_1 = require("../../../util/git/auth");
async function conanLockUpdate(conanFilePath, isLockFileMaintenance) {
    const command = `conan lock create ${(0, shlex_1.quote)(conanFilePath)}` +
        (isLockFileMaintenance ? ' --lockfile=""' : '');
    const execOptions = {
        extraEnv: { ...(0, auth_1.getGitEnvironmentVariables)(['conan']) },
        docker: {},
    };
    await (0, exec_1.exec)(command, execOptions);
}
async function updateArtifacts(updateArtifact) {
    const { packageFileName, updatedDeps, newPackageFileContent, config } = updateArtifact;
    logger_1.logger.trace(`conan.updateArtifacts(${packageFileName})`);
    const { isLockFileMaintenance } = config;
    if (updatedDeps.length === 0 && !isLockFileMaintenance) {
        logger_1.logger.trace('No conan.lock dependencies to update');
        return null;
    }
    const lockFileName = await (0, fs_1.findLocalSiblingOrParent)(packageFileName, 'conan.lock');
    if (!lockFileName) {
        logger_1.logger.trace('No conan.lock found');
        return null;
    }
    const existingLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
    if (!existingLockFileContent) {
        logger_1.logger.debug(`${lockFileName} read operation failed`);
        return null;
    }
    try {
        await (0, fs_1.writeLocalFile)(packageFileName, newPackageFileContent);
        logger_1.logger.trace(`Updating ${lockFileName}`);
        await conanLockUpdate(packageFileName, isLockFileMaintenance);
        const newLockFileContent = await (0, fs_1.readLocalFile)(lockFileName);
        if (!newLockFileContent) {
            logger_1.logger.debug(`New ${lockFileName} read operation failed`);
            return null;
        }
        if (existingLockFileContent === newLockFileContent) {
            logger_1.logger.trace(`${lockFileName} is unchanged`);
            return null;
        }
        logger_1.logger.trace(`Returning updated ${lockFileName}`);
        return [
            {
                file: {
                    type: 'addition',
                    path: lockFileName,
                    contents: newLockFileContent,
                },
            },
        ];
    }
    catch (err) {
        if (err.message === error_messages_1.TEMPORARY_ERROR) {
            throw err;
        }
        logger_1.logger.debug({ err, packageFileName, lockFileName }, 'Lockfile update failed');
        return [
            {
                artifactError: {
                    lockFile: lockFileName,
                    stderr: err.message,
                },
            },
        ];
    }
}
//# sourceMappingURL=artifacts.js.map