"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CargoLockSchemaToml = exports.CargoLockSchema = exports.CargoConfigSchema = exports.CargoManifestSchema = void 0;
const zod_1 = require("zod");
const schema_utils_1 = require("../../../util/schema-utils");
const crate_1 = require("../../datasource/crate");
const CargoDep = zod_1.z.union([
    zod_1.z
        .object({
        /** Path on disk to the crate sources */
        path: zod_1.z.string().optional(),
        /** Git URL for the dependency */
        git: zod_1.z.string().optional(),
        /** Semver version */
        version: zod_1.z.string().optional(),
        /** Name of a registry whose URL is configured in `.cargo/config.toml` or `.cargo/config` */
        registry: zod_1.z.string().optional(),
        /** Name of a package to look up */
        package: zod_1.z.string().optional(),
        /** Whether the dependency is inherited from the workspace */
        workspace: zod_1.z.boolean().optional(),
    })
        .transform(({ path, git, version, registry, package: pkg, workspace, }) => {
        let skipReason;
        let currentValue;
        let nestedVersion = false;
        if (version) {
            currentValue = version;
            nestedVersion = true;
        }
        else {
            currentValue = '';
            skipReason = 'invalid-dependency-specification';
        }
        if (path) {
            skipReason = 'path-dependency';
        }
        else if (git) {
            skipReason = 'git-dependency';
        }
        else if (workspace) {
            skipReason = 'inherited-dependency';
        }
        const dep = {
            currentValue,
            managerData: { nestedVersion },
            datasource: crate_1.CrateDatasource.id,
        };
        if (skipReason) {
            dep.skipReason = skipReason;
        }
        if (pkg) {
            dep.packageName = pkg;
        }
        if (registry) {
            dep.managerData.registryName = registry;
        }
        return dep;
    }),
    zod_1.z.string().transform((version) => ({
        currentValue: version,
        managerData: { nestedVersion: false },
        datasource: crate_1.CrateDatasource.id,
    })),
]);
const CargoDeps = zod_1.z.record(zod_1.z.string(), CargoDep).transform((record) => {
    const deps = [];
    for (const [depName, dep] of Object.entries(record)) {
        dep.depName = depName;
        deps.push(dep);
    }
    return deps;
});
const CargoSection = zod_1.z.object({
    dependencies: (0, schema_utils_1.withDepType)(CargoDeps, 'dependencies').optional(),
    'dev-dependencies': (0, schema_utils_1.withDepType)(CargoDeps, 'dev-dependencies').optional(),
    'build-dependencies': (0, schema_utils_1.withDepType)(CargoDeps, 'build-dependencies').optional(),
});
const CargoWorkspace = zod_1.z.object({
    dependencies: (0, schema_utils_1.withDepType)(CargoDeps, 'workspace.dependencies').optional(),
    package: zod_1.z
        .object({
        version: zod_1.z.string().optional(),
    })
        .optional(),
});
const CargoTarget = zod_1.z.record(zod_1.z.string(), CargoSection);
exports.CargoManifestSchema = schema_utils_1.Toml.pipe(CargoSection.extend({
    package: zod_1.z
        .object({
        version: zod_1.z
            .union([zod_1.z.string(), zod_1.z.object({ workspace: zod_1.z.literal(true) })])
            .optional(),
    })
        .optional(),
    workspace: CargoWorkspace.optional(),
    target: CargoTarget.optional(),
}));
const CargoConfigRegistry = zod_1.z.object({
    index: zod_1.z.string().optional(),
});
const CargoConfigSource = zod_1.z.object({
    'replace-with': zod_1.z.string().optional(),
    registry: zod_1.z.string().optional(),
});
exports.CargoConfigSchema = schema_utils_1.Toml.pipe(zod_1.z.object({
    registries: zod_1.z.record(zod_1.z.string(), CargoConfigRegistry).optional(),
    source: zod_1.z.record(zod_1.z.string(), CargoConfigSource).optional(),
}));
const CargoLockPackageSchema = zod_1.z.object({
    name: zod_1.z.string(),
    version: zod_1.z.string(),
    source: zod_1.z.string().optional(),
});
exports.CargoLockSchema = zod_1.z.object({
    package: zod_1.z.array(CargoLockPackageSchema).optional(),
});
exports.CargoLockSchemaToml = schema_utils_1.Toml.pipe(exports.CargoLockSchema);
//# sourceMappingURL=schema.js.map