"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.supportedRulesRegex = void 0;
exports.extractDepsFromFragmentData = extractDepsFromFragmentData;
exports.extractDepsFromFragment = extractDepsFromFragment;
exports.extract = extract;
const zod_1 = require("zod");
const regex_1 = require("../../../../util/regex");
const docker_1 = require("./docker");
const git_1 = require("./git");
const go_1 = require("./go");
const http_1 = require("./http");
const maven_1 = require("./maven");
const oci_1 = require("./oci");
const Target = zod_1.z.union([
    docker_1.DockerTarget,
    oci_1.OciTarget,
    git_1.GitTarget,
    go_1.GoTarget,
    http_1.HttpTarget,
    maven_1.MavenTarget,
]);
/**
 * Gather all rule names supported by Renovate in order to speed up parsing
 * by filtering out other syntactically correct rules we don't support yet.
 */
const supportedRules = [
    ...docker_1.dockerRules,
    ...oci_1.ociRules,
    ...git_1.gitRules,
    ...go_1.goRules,
    ...http_1.httpRules,
    ...maven_1.mavenRules,
];
exports.supportedRulesRegex = (0, regex_1.regEx)(`^(?:${supportedRules.join('|')})$`);
function extractDepsFromFragmentData(fragmentData) {
    const res = Target.safeParse(fragmentData);
    if (!res.success) {
        return [];
    }
    return res.data;
}
function extractDepsFromFragment(fragment) {
    const fragmentData = extract(fragment);
    return extractDepsFromFragmentData(fragmentData);
}
function extract(fragment) {
    if (fragment.type === 'string') {
        return fragment.value;
    }
    if (fragment.type === 'record') {
        const { children } = fragment;
        const result = {};
        for (const [key, value] of Object.entries(children)) {
            result[key] = extract(value);
        }
        return result;
    }
    return fragment.children.map(extract);
}
//# sourceMappingURL=index.js.map