"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpTarget = exports.httpRules = void 0;
exports.parseGithubPath = parseGithubPath;
exports.parseArchiveUrl = parseArchiveUrl;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const zod_1 = require("zod");
const regex_1 = require("../../../../util/regex");
const url_1 = require("../../../../util/url");
const github_releases_1 = require("../../../datasource/github-releases");
const github_tags_1 = require("../../../datasource/github-tags");
const gitlab_releases_1 = require("../../../datasource/gitlab-releases");
const gitlab_tags_1 = require("../../../datasource/gitlab-tags");
// Source: https://bazel.build/rules/lib/repo/http
const archives = [
    '.zip',
    '.tar',
    '.jar',
    '.war',
    '.aar',
    '.ar',
    '.deb',
    '.gz',
    '.tar.gz',
    '.tgz',
    '.bz2',
    '.tar.bz2',
    '.tbz2',
    '.xz',
    '.tar.xz',
    '.txz',
    '.zst',
    '.tar.zst',
    '.tzst',
];
const archiveSuffixRegex = (0, regex_1.regEx)(`(?:${archives.map(regex_1.escapeRegExp).join('|')})$`);
function stripArchiveSuffix(value) {
    return value.replace(archiveSuffixRegex, '');
}
function isHash(value) {
    return is_1.default.string(value) && (0, regex_1.regEx)(/[0-9a-z]{40}/i).test(value);
}
function parseGithubPath(pathname) {
    const [p0, p1, p2, p3, p4, p5] = pathname.split('/').slice(1);
    const packageName = p0 + '/' + p1;
    let datasource = '';
    let value = null;
    if (p2 === 'releases' && p3 === 'download') {
        // https://github.com/foo/bar/releases/download/1.2.3/bar-1.2.3.tar.gz
        datasource = github_releases_1.GithubReleasesDatasource.id;
        value = p4;
    }
    else if (p2 === 'archive' && p3 === 'refs' && p4 === 'tags') {
        // https://github.com/foo/bar/archive/refs/tags/v1.2.3.tar.gz
        datasource = github_tags_1.GithubTagsDatasource.id;
        value = p5;
    }
    else if (p2 === 'archive') {
        // https://github.com/foo/bar/archive/1.2.3.tar.gz
        datasource = github_tags_1.GithubTagsDatasource.id;
        value = p3;
    }
    if (!value) {
        return null;
    }
    value = stripArchiveSuffix(value);
    return isHash(value)
        ? { datasource, packageName, currentDigest: value }
        : { datasource, packageName, currentValue: value };
}
function parseGitlabPath(pathname) {
    // https://gitlab.com/libeigen/eigen/-/archive/3.3.5/eigen-3.3.5.zip
    // https://gitlab.com/libeigen/eigen/-/archive/90ee821c563fa20db4d64d6991ddca256d5c52f2/eigen-90ee821c563fa20db4d64d6991ddca256d5c52f2.tar.gz
    const [p0, p1, p2, p3, p4] = pathname.split('/').slice(1);
    const packageName = p0 + '/' + p1;
    if (p2 === '-' && p3 === 'archive' && p4) {
        return isHash(p4)
            ? {
                datasource: gitlab_tags_1.GitlabTagsDatasource.id,
                packageName,
                currentDigest: p4,
            }
            : {
                datasource: gitlab_releases_1.GitlabReleasesDatasource.id,
                packageName,
                currentValue: p4,
            };
    }
    return null;
}
function parseArchiveUrl(urlString) {
    if (!urlString) {
        return null;
    }
    const url = (0, url_1.parseUrl)(urlString);
    if (url?.host === 'github.com') {
        return parseGithubPath(url.pathname);
    }
    if (url?.host === 'gitlab.com') {
        return parseGitlabPath(url.pathname);
    }
    return null;
}
exports.httpRules = [
    'http_archive',
    '_http_archive',
    'http_file',
    '_http_file',
];
exports.HttpTarget = zod_1.z
    .object({
    rule: zod_1.z.enum(exports.httpRules),
    name: zod_1.z.string(),
    url: zod_1.z.string().optional(),
    urls: zod_1.z.array(zod_1.z.string()).optional(),
    sha256: zod_1.z.string(),
})
    .refine(({ url, urls }) => !!url || !!urls)
    .transform(({ rule, name, url, urls = [] }) => {
    const parsedUrl = [url, ...urls].map(parseArchiveUrl).find(is_1.default.truthy);
    if (!parsedUrl) {
        return [];
    }
    const dep = {
        datasource: parsedUrl.datasource,
        depType: rule,
        depName: name,
        packageName: parsedUrl.packageName,
    };
    // We don't want to set both `currentValue` and `currentDigest`.
    //
    // What we want is to provide the first occurrence of `currentValue`,
    // or, if it's not available, `currentDigest`.
    //
    // Auto-replace mechanism will replace this first occurrence,
    // and artifact update function will do the rest.
    //
    // Hence, `if-else-if` is being used here.
    if (parsedUrl.currentValue) {
        dep.currentValue = parsedUrl.currentValue;
    }
    else if (parsedUrl.currentDigest) {
        dep.currentDigest = parsedUrl.currentDigest;
    }
    return [dep];
});
//# sourceMappingURL=http.js.map