"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitTarget = exports.gitRules = void 0;
const tslib_1 = require("tslib");
const github_url_from_git_1 = tslib_1.__importDefault(require("github-url-from-git"));
const zod_1 = require("zod");
const logger_1 = require("../../../../logger");
const regex_1 = require("../../../../util/regex");
const url_1 = require("../../../../util/url");
const github_releases_1 = require("../../../datasource/github-releases");
const github_tags_1 = require("../../../datasource/github-tags");
const githubUrlRegex = (0, regex_1.regEx)(/^https:\/\/github\.com\/(?<packageName>[^/]+\/[^/]+)/);
function githubPackageName(input) {
    // istanbul ignore if
    if (!(0, url_1.isHttpUrl)(input)) {
        logger_1.logger.once.info({ url: input }, `Bazel: non-https git_repository URL`);
    }
    return (0, github_url_from_git_1.default)(input)?.match(githubUrlRegex)?.groups?.packageName;
}
exports.gitRules = [
    'git_repository',
    '_git_repository',
    'new_git_repository',
    '_new_git_repository',
];
exports.GitTarget = zod_1.z
    .object({
    rule: zod_1.z.enum(exports.gitRules),
    name: zod_1.z.string(),
    tag: zod_1.z.string().optional(),
    commit: zod_1.z.string().optional(),
    remote: zod_1.z.string(),
})
    .refine(({ tag, commit }) => !!tag || !!commit)
    .transform(({ rule, name, tag, commit, remote }) => {
    const dep = {
        depType: rule,
        depName: name,
    };
    if (tag) {
        dep.currentValue = tag;
    }
    if (commit) {
        dep.currentDigest = commit;
    }
    const githubPackage = githubPackageName(remote);
    if (githubPackage) {
        dep.packageName = githubPackage;
        if (dep.currentValue) {
            dep.datasource = github_releases_1.GithubReleasesDatasource.id;
        }
        else {
            dep.datasource = github_tags_1.GithubTagsDatasource.id;
        }
    }
    if (!dep.datasource) {
        dep.skipReason = 'unsupported-datasource';
    }
    return [dep];
});
//# sourceMappingURL=git.js.map