"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommandEntry = exports.BazelOption = exports.ImportEntry = void 0;
exports.expandWorkspacePath = expandWorkspacePath;
exports.sanitizeOptions = sanitizeOptions;
exports.parse = parse;
exports.read = read;
const tslib_1 = require("tslib");
const upath_1 = tslib_1.__importDefault(require("upath"));
const logger_1 = require("../../../logger");
const array_1 = require("../../../util/array");
const fs = tslib_1.__importStar(require("../../../util/fs"));
const regex_1 = require("../../../util/regex");
const importRegex = (0, regex_1.regEx)(`^(?<type>(?:try-)?import)\\s+(?<path>\\S+)$`);
const optionRegex = (0, regex_1.regEx)(`^(?<command>\\w+)(:(?<config>\\S+))?\\s+(?<options>.*)$`);
const spaceRegex = (0, regex_1.regEx)(`\\s+`);
class ImportEntry {
    path;
    isTry;
    entryType = 'import';
    constructor(path, isTry) {
        this.path = path;
        this.isTry = isTry;
    }
}
exports.ImportEntry = ImportEntry;
class BazelOption {
    name;
    value;
    constructor(name, value) {
        this.name = name;
        this.value = value;
    }
    static parse(input) {
        const options = [];
        const parts = input.split(spaceRegex);
        for (let i = 0; i < parts.length; i++) {
            const part = parts[i];
            if (!part.startsWith('--')) {
                continue;
            }
            const nameStartIdx = 2;
            // Check for --option_name=option_value
            const equalSignIdx = part.indexOf('=');
            if (equalSignIdx >= 0) {
                const name = part.substring(nameStartIdx, equalSignIdx);
                const value = part.substring(equalSignIdx + 1);
                options.push(new BazelOption(name, value));
                continue;
            }
            const name = part.substring(nameStartIdx);
            const nextIdx = i + 1;
            // Check for --option_name OR --option_name option_value
            const value = nextIdx < parts.length && !parts[nextIdx].startsWith('--')
                ? parts[nextIdx]
                : undefined;
            options.push(new BazelOption(name, value));
        }
        return options;
    }
}
exports.BazelOption = BazelOption;
class CommandEntry {
    command;
    options;
    config;
    entryType = 'command';
    constructor(command, options, config) {
        this.command = command;
        this.options = options;
        this.config = config;
    }
    getOption(name) {
        return this.options.find((bo) => bo.name === name);
    }
}
exports.CommandEntry = CommandEntry;
function shouldProcessLine(line) {
    if (line.length === 0) {
        return false;
    }
    return !line.startsWith('#');
}
function createEntry(line) {
    const importResult = importRegex.exec(line);
    if (importResult?.groups) {
        const irGroups = importResult.groups;
        return new ImportEntry(irGroups.path, irGroups.type === 'try-import');
    }
    const optionResult = optionRegex.exec(line);
    if (optionResult?.groups) {
        const orGroups = optionResult.groups;
        return new CommandEntry(orGroups.command, BazelOption.parse(orGroups.options), orGroups.config);
    }
    return undefined;
}
function expandWorkspacePath(value, workspaceDir) {
    if (!value.includes('%workspace%')) {
        return value;
    }
    const absolutePath = upath_1.default.resolve(workspaceDir);
    const expandedPath = value.replace('%workspace%', absolutePath);
    if (!fs.isValidLocalPath(expandedPath)) {
        return null;
    }
    return expandedPath;
}
function sanitizeOptions(options, workspaceDir) {
    return options
        .map((option) => {
        if (!option.value) {
            return option;
        }
        const expandedPath = expandWorkspacePath(option.value, workspaceDir);
        if (!expandedPath) {
            logger_1.logger.debug(`Skipping invalid workspace path: ${option.value} in ${workspaceDir}`);
            return null;
        }
        return new BazelOption(option.name, expandedPath);
    })
        .filter(array_1.isNotNullOrUndefined);
}
function parse(contents) {
    return contents
        .split('\n')
        .map((l) => l.trim())
        .filter(shouldProcessLine)
        .map(createEntry)
        .filter(array_1.isNotNullOrUndefined);
}
async function readFile(file, workspaceDir, readFiles) {
    if (readFiles.has(file)) {
        throw new Error(`Attempted to read a bazelrc multiple times. file: ${file}`);
    }
    readFiles.add(file);
    const contents = await fs.readLocalFile(file, 'utf8');
    if (!contents) {
        return [];
    }
    const entries = parse(contents);
    const results = [];
    for (const entry of entries) {
        if (entry.entryType === 'command') {
            const sanitizedOptions = sanitizeOptions(entry.options, workspaceDir);
            results.push(new CommandEntry(entry.command, sanitizedOptions, entry.config));
            continue;
        }
        const importFile = upath_1.default.normalize(entry.path.replace('%workspace%', workspaceDir));
        if (fs.isValidLocalPath(importFile)) {
            const importEntries = await readFile(importFile, workspaceDir, readFiles);
            results.push(...importEntries);
        }
        else {
            logger_1.logger.debug(`Skipping non-local .bazelrc import ${importFile}`);
        }
    }
    return results;
}
async function read(workspaceDir) {
    const bazelrcPath = upath_1.default.join(workspaceDir, '.bazelrc');
    const readFiles = new Set();
    return await readFile(bazelrcPath, workspaceDir, readFiles);
}
//# sourceMappingURL=bazelrc.js.map