"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SbtPluginDatasource = exports.SBT_PLUGINS_REPO = void 0;
const tslib_1 = require("tslib");
const xmldoc_1 = require("xmldoc");
const logger_1 = require("../../../logger");
const http_1 = require("../../../util/http");
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const ivyVersioning = tslib_1.__importStar(require("../../versioning/ivy"));
const compare_1 = require("../../versioning/maven/compare");
const datasource_1 = require("../datasource");
const common_1 = require("../maven/common");
const util_1 = require("../maven/util");
const util_2 = require("../sbt-package/util");
exports.SBT_PLUGINS_REPO = 'https://repo.scala-sbt.org/scalasbt/sbt-plugin-releases';
class SbtPluginDatasource extends datasource_1.Datasource {
    static id = 'sbt-plugin';
    defaultRegistryUrls = [exports.SBT_PLUGINS_REPO, common_1.MAVEN_REPO];
    defaultVersioning = ivyVersioning.id;
    registryStrategy = 'merge';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined from the `scm` tags in the results.';
    constructor() {
        super(SbtPluginDatasource.id);
        this.http = new http_1.Http('sbt');
    }
    // istanbul ignore next: to be rewritten
    async getArtifactSubdirs(searchRoot, artifact, scalaVersion) {
        const pkgUrl = (0, url_1.ensureTrailingSlash)(searchRoot);
        const indexContent = await (0, util_1.downloadHttpContent)(this.http, pkgUrl);
        if (indexContent) {
            const rootPath = new URL(pkgUrl).pathname;
            let artifactSubdirs = (0, util_2.extractPageLinks)(indexContent, (href) => {
                const path = href.replace(rootPath, '');
                if (path.startsWith(`${artifact}_native`) ||
                    path.startsWith(`${artifact}_sjs`)) {
                    return null;
                }
                if (path === artifact || path.startsWith(`${artifact}_`)) {
                    return path;
                }
                return null;
            });
            if (scalaVersion &&
                artifactSubdirs.includes(`${artifact}_${scalaVersion}`)) {
                artifactSubdirs = [`${artifact}_${scalaVersion}`];
            }
            return artifactSubdirs;
        }
        return null;
    }
    // istanbul ignore next: to be rewritten
    async getPackageReleases(searchRoot, artifactSubdirs) {
        if (artifactSubdirs) {
            const releases = [];
            for (const searchSubdir of artifactSubdirs) {
                const pkgUrl = (0, url_1.ensureTrailingSlash)(`${searchRoot}/${searchSubdir}`);
                const content = await (0, util_1.downloadHttpContent)(this.http, pkgUrl);
                if (content) {
                    const rootPath = new URL(pkgUrl).pathname;
                    const subdirReleases = (0, util_2.extractPageLinks)(content, (href) => {
                        const path = href.replace(rootPath, '');
                        if (path.startsWith('.')) {
                            return null;
                        }
                        return path;
                    });
                    subdirReleases.forEach((x) => releases.push(x));
                }
            }
            if (releases.length) {
                return [...new Set(releases)].sort(compare_1.compare);
            }
        }
        return null;
    }
    // istanbul ignore next: to be rewritten
    async getUrls(searchRoot, artifactDirs, version) {
        const result = {};
        if (!artifactDirs?.length) {
            return result;
        }
        if (!version) {
            return result;
        }
        for (const artifactDir of artifactDirs) {
            const [artifact] = artifactDir.split('_');
            const pomFileNames = [
                `${artifactDir}-${version}.pom`,
                `${artifact}-${version}.pom`,
            ];
            for (const pomFileName of pomFileNames) {
                const pomUrl = `${searchRoot}/${artifactDir}/${version}/${pomFileName}`;
                const content = await (0, util_1.downloadHttpContent)(this.http, pomUrl);
                if (content) {
                    const pomXml = new xmldoc_1.XmlDocument(content);
                    const homepage = pomXml.valueWithPath('url');
                    if (homepage) {
                        result.homepage = homepage;
                    }
                    const sourceUrl = pomXml.valueWithPath('scm.url');
                    if (sourceUrl) {
                        result.sourceUrl = sourceUrl
                            .replace((0, regex_1.regEx)(/^scm:/), '')
                            .replace((0, regex_1.regEx)(/^git:/), '')
                            .replace((0, regex_1.regEx)(/^git@github.com:/), 'https://github.com/')
                            .replace((0, regex_1.regEx)(/\.git$/), '');
                    }
                    return result;
                }
            }
        }
        return result;
    }
    async resolvePluginReleases(rootUrl, artifact, scalaVersion) {
        const searchRoot = `${rootUrl}/${artifact}`;
        const hrefFilterMap = (href) => {
            if (href.startsWith('.')) {
                return null;
            }
            return href;
        };
        const searchRootContent = await (0, util_1.downloadHttpContent)(this.http, (0, url_1.ensureTrailingSlash)(searchRoot));
        if (searchRootContent) {
            const releases = [];
            const scalaVersionItems = (0, util_2.extractPageLinks)(searchRootContent, hrefFilterMap);
            const scalaVersions = scalaVersionItems.map((x) => x.replace((0, regex_1.regEx)(/^scala_/), ''));
            const searchVersions = scalaVersions.includes(scalaVersion)
                ? [scalaVersion]
                : scalaVersions;
            for (const searchVersion of searchVersions) {
                const searchSubRoot = `${searchRoot}/scala_${searchVersion}`;
                const subRootContent = await (0, util_1.downloadHttpContent)(this.http, (0, url_1.ensureTrailingSlash)(searchSubRoot));
                if (subRootContent) {
                    const sbtVersionItems = (0, util_2.extractPageLinks)(subRootContent, hrefFilterMap);
                    for (const sbtItem of sbtVersionItems) {
                        const releasesRoot = `${searchSubRoot}/${sbtItem}`;
                        const releasesIndexContent = await (0, util_1.downloadHttpContent)(this.http, (0, url_1.ensureTrailingSlash)(releasesRoot));
                        if (releasesIndexContent) {
                            const releasesParsed = (0, util_2.extractPageLinks)(releasesIndexContent, hrefFilterMap);
                            releasesParsed.forEach((x) => releases.push(x));
                        }
                    }
                }
            }
            if (releases.length) {
                return [...new Set(releases)].sort(compare_1.compare);
            }
        }
        return null;
    }
    async getReleases({ packageName, registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const [groupId, artifactId] = packageName.split(':');
        const groupIdSplit = groupId.split('.');
        const artifactIdSplit = artifactId.split('_');
        const [artifact, scalaVersion] = artifactIdSplit;
        const repoRoot = (0, url_1.ensureTrailingSlash)(registryUrl);
        const searchRoots = [];
        // Optimize lookup order
        if (!registryUrl.startsWith(common_1.MAVEN_REPO)) {
            searchRoots.push(`${repoRoot}${groupIdSplit.join('.')}`);
        }
        searchRoots.push(`${repoRoot}${groupIdSplit.join('/')}`);
        for (const searchRoot of searchRoots) {
            let versions = await this.resolvePluginReleases(searchRoot, artifact, scalaVersion);
            let urls = {};
            if (!versions?.length) {
                const artifactSubdirs = await this.getArtifactSubdirs(searchRoot, artifact, scalaVersion);
                versions = await this.getPackageReleases(searchRoot, artifactSubdirs);
                const latestVersion = (0, util_2.getLatestVersion)(versions);
                urls = await this.getUrls(searchRoot, artifactSubdirs, latestVersion);
            }
            const dependencyUrl = `${searchRoot}/${artifact}`;
            logger_1.logger.trace({ dependency: packageName, versions }, `Package versions`);
            if (versions) {
                return {
                    ...urls,
                    dependencyUrl,
                    releases: versions.map((v) => ({ version: v })),
                };
            }
        }
        logger_1.logger.debug(`No versions found for ${packageName} in ${searchRoots.length} repositories`);
        return null;
    }
}
exports.SbtPluginDatasource = SbtPluginDatasource;
//# sourceMappingURL=index.js.map