"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PuppetForgeDatasource = void 0;
const timestamp_1 = require("../../../util/timestamp");
const datasource_1 = require("../datasource");
const common_1 = require("./common");
class PuppetForgeDatasource extends datasource_1.Datasource {
    static id = 'puppet-forge';
    constructor() {
        super(PuppetForgeDatasource.id);
    }
    defaultRegistryUrls = [common_1.PUPPET_FORGE];
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `created_at` field from the response.';
    async getReleases({ packageName, registryUrl, }) {
        // https://forgeapi.puppet.com
        const moduleSlug = packageName.replace('/', '-');
        // TODO: types (#22198)
        const url = `${registryUrl}/v3/modules/${moduleSlug}?exclude_fields=current_release`;
        let module;
        try {
            const response = await this.http.getJsonUnchecked(url);
            module = response.body;
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        const releases = module?.releases?.map((release) => ({
            version: release.version,
            downloadUrl: release.file_uri,
            releaseTimestamp: (0, timestamp_1.asTimestamp)(release.created_at),
            registryUrl,
        }));
        if (!releases?.length) {
            return null;
        }
        return PuppetForgeDatasource.createReleaseResult(releases, module);
    }
    static createReleaseResult(releases, module) {
        const result = {
            releases,
            // the homepage url in the fixtures is a github repo, we can use this as sourceUrl
            homepage: module.homepage_url,
        };
        if (module.deprecated_for) {
            result.deprecationMessage = module.deprecated_for;
        }
        return result;
    }
}
exports.PuppetForgeDatasource = PuppetForgeDatasource;
//# sourceMappingURL=index.js.map