"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NugetV2Api = void 0;
const xmldoc_1 = require("xmldoc");
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const timestamp_1 = require("../../../util/timestamp");
const common_1 = require("./common");
class NugetV2Api {
    getPkgProp(pkgInfo, propName) {
        return pkgInfo.childNamed('m:properties')?.childNamed(`d:${propName}`)?.val;
    }
    async getReleases(http, feedUrl, pkgName) {
        const dep = {
            releases: [],
        };
        let pkgUrlList = `${feedUrl.replace((0, regex_1.regEx)(/\/+$/), '')}/FindPackagesById()?id=%27${pkgName}%27&$select=Version,IsLatestVersion,ProjectUrl,Published`;
        while (pkgUrlList !== null) {
            // typescript issue
            const pkgVersionsListRaw = await http.getText(pkgUrlList);
            const pkgVersionsListDoc = new xmldoc_1.XmlDocument(pkgVersionsListRaw.body);
            const pkgInfoList = pkgVersionsListDoc.childrenNamed('entry');
            for (const pkgInfo of pkgInfoList) {
                const version = this.getPkgProp(pkgInfo, 'Version');
                const releaseTimestamp = (0, timestamp_1.asTimestamp)(this.getPkgProp(pkgInfo, 'Published'));
                dep.releases.push({
                    // TODO: types (#22198)
                    version: (0, common_1.removeBuildMeta)(`${version}`),
                    releaseTimestamp,
                });
                try {
                    const pkgIsLatestVersion = this.getPkgProp(pkgInfo, 'IsLatestVersion');
                    if (pkgIsLatestVersion === 'true') {
                        dep.tags = { latest: (0, common_1.removeBuildMeta)(`${version}`) };
                        const projectUrl = this.getPkgProp(pkgInfo, 'ProjectUrl');
                        if (projectUrl) {
                            dep.sourceUrl = (0, common_1.massageUrl)(projectUrl);
                        }
                    }
                }
                catch (err) /* istanbul ignore next */ {
                    logger_1.logger.debug({ err, pkgName, feedUrl }, `nuget registry failure: can't parse pkg info for project url`);
                }
            }
            const nextPkgUrlListLink = pkgVersionsListDoc
                .childrenNamed('link')
                .find((node) => node.attr.rel === 'next');
            pkgUrlList = nextPkgUrlListLink ? nextPkgUrlListLink.attr.href : null;
        }
        // dep not found if no release, so we can try next registry
        if (dep.releases.length === 0) {
            return null;
        }
        return dep;
    }
}
exports.NugetV2Api = NugetV2Api;
//# sourceMappingURL=v2.js.map