"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeBuildMeta = removeBuildMeta;
exports.massageUrl = massageUrl;
exports.parseRegistryUrl = parseRegistryUrl;
exports.sortNugetVersions = sortNugetVersions;
const logger_1 = require("../../../logger");
const regex_1 = require("../../../util/regex");
const url_1 = require("../../../util/url");
const nuget_1 = require("../../versioning/nuget");
const buildMetaRe = (0, regex_1.regEx)(/\+.+$/g);
function removeBuildMeta(version) {
    return version.replace(buildMetaRe, '');
}
const urlWhitespaceRe = (0, regex_1.regEx)(/\s/g);
function massageUrl(url) {
    if (url === null || url === undefined) {
        return null;
    }
    let resultUrl = url;
    // During `dotnet pack` certain URLs are being URL decoded which may introduce whitespace
    // and causes Markdown link generation problems.
    resultUrl = resultUrl.replace(urlWhitespaceRe, '%20');
    return resultUrl;
}
const protocolVersionRegExp = (0, regex_1.regEx)(/#protocolVersion=(?<protocol>2|3)/);
function parseRegistryUrl(registryUrl) {
    const parsedUrl = (0, url_1.parseUrl)(registryUrl);
    if (!parsedUrl) {
        logger_1.logger.debug({ urL: registryUrl }, `nuget registry failure: can't parse ${registryUrl}`);
        return { feedUrl: registryUrl, protocolVersion: null };
    }
    let protocolVersion = 2;
    const protocolVersionMatch = protocolVersionRegExp.exec(parsedUrl.hash)?.groups;
    if (protocolVersionMatch) {
        const { protocol } = protocolVersionMatch;
        parsedUrl.hash = '';
        protocolVersion = Number.parseInt(protocol, 10);
    }
    else if (parsedUrl.pathname.endsWith('.json')) {
        protocolVersion = 3;
    }
    const feedUrl = parsedUrl.href;
    return { feedUrl, protocolVersion };
}
/**
 * Compare two versions. Return:
 * - `1` if `a > b` or `b` is invalid
 * - `-1` if `a < b` or `a` is invalid
 * - `0` if `a == b` or both `a` and `b` are invalid
 */
function sortNugetVersions(a, b) {
    if (nuget_1.api.isValid(a)) {
        if (nuget_1.api.isValid(b)) {
            return nuget_1.api.sortVersions(a, b);
        }
        else {
            return 1;
        }
    }
    else if (nuget_1.api.isValid(b)) {
        return -1;
    }
    else {
        return 0;
    }
}
//# sourceMappingURL=common.js.map