"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HexpmBobDatasource = void 0;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const logger_1 = require("../../../logger");
const external_host_error_1 = require("../../../types/errors/external-host-error");
const decorator_1 = require("../../../util/cache/package/decorator");
const http_1 = require("../../../util/http");
const regex_1 = require("../../../util/regex");
const timestamp_1 = require("../../../util/timestamp");
const semver_1 = require("../../versioning/semver");
const datasource_1 = require("../datasource");
const common_1 = require("./common");
class HexpmBobDatasource extends datasource_1.Datasource {
    static id = common_1.datasource;
    constructor() {
        super(common_1.datasource);
    }
    customRegistrySupport = true;
    defaultRegistryUrls = [common_1.defaultRegistryUrl];
    caching = true;
    defaultVersioning = semver_1.id;
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `buildDate` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL https://github.com/elixir-lang/elixir.git for the `elixir` package and the https://github.com/erlang/otp.git URL for the `erlang` package.';
    async getReleases({ registryUrl, packageName, }) {
        const packageType = HexpmBobDatasource.getPackageType(packageName);
        if (!packageType) {
            return null;
        }
        logger_1.logger.trace({ registryUrl, packageName }, `fetching hex.pm bob ${packageName} release`);
        const url = `${registryUrl}/builds/${packageName}/builds.txt`;
        const result = {
            releases: [],
            ...HexpmBobDatasource.getPackageDetails(packageType),
        };
        try {
            const { body } = await this.http.getText(url);
            result.releases = body
                .split('\n')
                .map((line) => line.trim())
                .filter(is_1.default.nonEmptyString)
                .map((line) => {
                const [version, gitRef, buildDate] = line.split(' ');
                return {
                    gitRef,
                    isStable: HexpmBobDatasource.isStable(version, packageType),
                    releaseTimestamp: (0, timestamp_1.asTimestamp)(buildDate),
                    version: HexpmBobDatasource.cleanVersion(version, packageType),
                };
            });
        }
        catch (err) {
            if (err instanceof http_1.HttpError && err.response?.statusCode !== 404) {
                throw new external_host_error_1.ExternalHostError(err);
            }
            this.handleGenericErrors(err);
        }
        return result.releases.length > 0 ? result : null;
    }
    static getPackageType(packageName) {
        if (packageName === 'elixir') {
            return 'elixir';
        }
        if (/^otp\/\w+-\d+\.\d+$/.test(packageName)) {
            return 'erlang';
        }
        return null;
    }
    static cleanVersion(version, packageType) {
        switch (packageType) {
            case 'elixir':
                return version.replace(/^v/, '');
            case 'erlang':
                return version.replace(/^OTP-/, '');
        }
    }
    static isStable(version, packageType) {
        switch (packageType) {
            case 'elixir':
                return (0, regex_1.regEx)(/^v\d+\.\d+\.\d+($|-otp)/).test(version);
            case 'erlang':
                return version.startsWith('OTP-');
        }
    }
    static getPackageDetails(packageType) {
        switch (packageType) {
            case 'elixir':
                return {
                    homepage: 'https://elixir-lang.org/',
                    sourceUrl: 'https://github.com/elixir-lang/elixir.git',
                };
            case 'erlang':
                return {
                    homepage: 'https://www.erlang.org/',
                    sourceUrl: 'https://github.com/erlang/otp.git',
                };
        }
    }
}
exports.HexpmBobDatasource = HexpmBobDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${common_1.datasource}`,
        key: ({ registryUrl, packageName }) => `${registryUrl}:${packageName}`,
    })
], HexpmBobDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map