"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HelmRepositorySchema = void 0;
const zod_1 = require("zod");
const common_1 = require("../../../util/common");
const url_1 = require("../../../util/git/url");
const regex_1 = require("../../../util/regex");
const schema_utils_1 = require("../../../util/schema-utils");
const timestamp_1 = require("../../../util/timestamp");
const HelmReleaseSchema = zod_1.z.object({
    version: zod_1.z.string(),
    created: timestamp_1.MaybeTimestamp,
    digest: zod_1.z.string().optional().catch(undefined),
    home: zod_1.z.string().optional().catch(undefined),
    sources: zod_1.z.array(zod_1.z.string()).catch([]),
    urls: zod_1.z.array(zod_1.z.string()).catch([]),
});
const chartRepo = (0, regex_1.regEx)(/charts?|helm|helm-charts/i);
function isPossibleChartRepo(url) {
    if ((0, common_1.detectPlatform)(url) === null) {
        return false;
    }
    const parsed = (0, url_1.parseGitUrl)(url);
    return chartRepo.test(parsed.name);
}
const githubRelease = (0, regex_1.regEx)(/^(https:\/\/github\.com\/[^/]+\/[^/]+)\/releases\//);
function getSourceUrl(release) {
    // it's a github release :)
    const [githubUrl] = release.urls;
    const releaseMatch = githubRelease.exec(githubUrl);
    if (releaseMatch) {
        return releaseMatch[1];
    }
    if (release.home && isPossibleChartRepo(release.home)) {
        return release.home;
    }
    for (const url of release.sources) {
        if (isPossibleChartRepo(url)) {
            return url;
        }
    }
    // fallback
    return release.sources[0];
}
exports.HelmRepositorySchema = zod_1.z
    .object({
    entries: (0, schema_utils_1.LooseRecord)(zod_1.z.string(), HelmReleaseSchema.array()
        .min(1)
        .transform((helmReleases) => {
        const latestRelease = helmReleases[0];
        const homepage = latestRelease.home;
        const sourceUrl = getSourceUrl(latestRelease);
        const releases = helmReleases.map(({ version, created: releaseTimestamp, digest: newDigest, }) => ({
            version,
            releaseTimestamp,
            newDigest,
        }));
        return { homepage, sourceUrl, releases };
    })),
})
    .transform(({ entries }) => entries);
//# sourceMappingURL=schema.js.map