"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GradleVersionDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const regex_1 = require("../../../util/regex");
const timestamp_1 = require("../../../util/timestamp");
const gradleVersioning = tslib_1.__importStar(require("../../versioning/gradle"));
const datasource_1 = require("../datasource");
class GradleVersionDatasource extends datasource_1.Datasource {
    static id = 'gradle-version';
    constructor() {
        super(GradleVersionDatasource.id);
    }
    defaultRegistryUrls = [
        'https://services.gradle.org/versions/all',
    ];
    defaultVersioning = gradleVersioning.id;
    registryStrategy = 'merge';
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `buildTime` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'We use the URL: https://github.com/gradle/gradle.';
    async getReleases({ registryUrl, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        let releases;
        try {
            const response = await this.http.getJsonUnchecked(registryUrl);
            releases = response.body
                .filter((release) => !release.snapshot && !release.nightly)
                .map((release) => {
                const { version, buildTime } = release;
                const gitRef = GradleVersionDatasource.getGitRef(release.version);
                const releaseTimestamp = (0, timestamp_1.asTimestamp)(buildTime);
                const result = { version, gitRef, releaseTimestamp };
                if (release.broken) {
                    result.isDeprecated = true;
                }
                return result;
            });
        }
        catch (err) {
            this.handleGenericErrors(err);
        }
        const res = {
            releases,
            homepage: 'https://gradle.org',
            sourceUrl: 'https://github.com/gradle/gradle',
        };
        if (res.releases.length) {
            return res;
        }
        return null;
    }
    /**
     * Calculate `gitTag` based on `version`:
     *   - `8.1.2` -> `v8.1.2`
     *   - `8.2` -> `v8.2.0`
     *   - `8.2-rc-1` -> `v8.2.0-RC1`
     *   - `8.2-milestone-1` -> `v8.2.0-M1`
     */
    static getGitRef(version) {
        const [versionPart, typePart, unstablePart] = version.split((0, regex_1.regEx)(/-([a-z]+)-/));
        let suffix = '';
        if (typePart === 'rc') {
            suffix = `-RC${unstablePart}`;
        }
        else if (typePart === 'milestone') {
            suffix = `-M${unstablePart}`;
        }
        const [major, minor, patch = '0'] = versionPart.split('.');
        return `v${major}.${minor}.${patch}${suffix}`;
    }
}
exports.GradleVersionDatasource = GradleVersionDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${GradleVersionDatasource.id}`,
        // TODO: types (#22198)
        key: ({ registryUrl }) => `${registryUrl}`,
    })
], GradleVersionDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map