"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseGoproxy = parseGoproxy;
exports.parseNoproxy = parseNoproxy;
const tslib_1 = require("tslib");
const is_1 = tslib_1.__importDefault(require("@sindresorhus/is"));
const moo_1 = tslib_1.__importDefault(require("moo"));
const memCache = tslib_1.__importStar(require("../../../util/cache/memory"));
const env_1 = require("../../../util/env");
const regex_1 = require("../../../util/regex");
/**
 * Parse `GOPROXY` to the sequence of url + fallback strategy tags.
 *
 * @example
 * parseGoproxy('foo.example.com|bar.example.com,baz.example.com')
 * // [
 * //   { url: 'foo.example.com', fallback: '|' },
 * //   { url: 'bar.example.com', fallback: ',' },
 * //   { url: 'baz.example.com', fallback: '|' },
 * // ]
 *
 * @see https://golang.org/ref/mod#goproxy-protocol
 */
function parseGoproxy(input = (0, env_1.getEnv)().GOPROXY) {
    if (!is_1.default.string(input)) {
        return [];
    }
    const cacheKey = `goproxy::${input}`;
    const cachedResult = memCache.get(cacheKey);
    if (cachedResult) {
        return cachedResult;
    }
    const result = input
        .split((0, regex_1.regEx)(/([^,|]*(?:,|\|))/))
        .filter(Boolean)
        .map((s) => s.split(/(?=,|\|)/)) // TODO: #12872 lookahead
        .map(([url, separator]) => ({
        url,
        fallback: separator === ',' ? ',' : '|',
    }));
    memCache.set(cacheKey, result);
    return result;
}
// https://golang.org/pkg/path/#Match
const noproxyLexer = moo_1.default.states({
    main: {
        separator: {
            match: /\s*?,\s*?/, // TODO #12870
            value: (_) => '|',
        },
        asterisk: {
            match: '*',
            value: (_) => '[^/]*',
        },
        qmark: {
            match: '?',
            value: (_) => '[^/]',
        },
        characterRangeOpen: {
            match: '[',
            push: 'characterRange',
            value: (_) => '[',
        },
        trailingSlash: {
            match: /\/$/,
            value: (_) => '',
        },
        char: {
            match: /[^*?\\[\n]/,
            value: (s) => s.replace((0, regex_1.regEx)('\\.', 'g'), '\\.'),
        },
        escapedChar: {
            match: /\\./, // TODO #12870
            value: (s) => s.slice(1),
        },
    },
    characterRange: {
        char: /[^\\\]\n]/, // TODO #12870
        escapedChar: {
            match: /\\./, // TODO #12870
            value: (s) => s.slice(1),
        },
        characterRangeEnd: {
            match: ']',
            pop: 1,
        },
    },
});
function parseNoproxy(input = (() => {
    const env = (0, env_1.getEnv)();
    return env.GONOPROXY ?? env.GOPRIVATE;
})()) {
    if (!is_1.default.string(input)) {
        return null;
    }
    const cacheKey = `noproxy::${input}`;
    const cachedResult = memCache.get(cacheKey);
    if (cachedResult !== undefined) {
        return cachedResult;
    }
    const noproxyPattern = [...noproxyLexer.reset(input)]
        .map(({ value }) => value)
        .join('');
    const result = noproxyPattern
        ? (0, regex_1.regEx)(`^(?:${noproxyPattern})(?:/.*)?$`)
        : null;
    memCache.set(cacheKey, result);
    return result;
}
//# sourceMappingURL=goproxy-parser.js.map