"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabReleasesDatasource = void 0;
const tslib_1 = require("tslib");
const decorator_1 = require("../../../util/cache/package/decorator");
const gitlab_1 = require("../../../util/http/gitlab");
const timestamp_1 = require("../../../util/timestamp");
const datasource_1 = require("../datasource");
class GitlabReleasesDatasource extends datasource_1.Datasource {
    static id = 'gitlab-releases';
    defaultRegistryUrls = ['https://gitlab.com'];
    static registryStrategy = 'first';
    releaseTimestampSupport = true;
    releaseTimestampNote = 'The release timestamp is determined from the `released_at` field in the results.';
    sourceUrlSupport = 'package';
    sourceUrlNote = 'The source URL is determined by using the `packageName` and `registryUrl`.';
    constructor() {
        super(GitlabReleasesDatasource.id);
        this.http = new gitlab_1.GitlabHttp(GitlabReleasesDatasource.id);
    }
    async getReleases({ registryUrl, packageName, }) {
        /* v8 ignore next 3 -- should never happen */
        if (!registryUrl) {
            return null;
        }
        const urlEncodedRepo = encodeURIComponent(packageName);
        const apiUrl = `${registryUrl}/api/v4/projects/${urlEncodedRepo}/releases`;
        try {
            const gitlabReleasesResponse = (await this.http.getJsonUnchecked(apiUrl)).body;
            return {
                sourceUrl: `${registryUrl}/${packageName}`,
                releases: gitlabReleasesResponse.map(({ tag_name, released_at }) => {
                    const release = {
                        registryUrl,
                        gitRef: tag_name,
                        version: tag_name,
                        releaseTimestamp: (0, timestamp_1.asTimestamp)(released_at),
                    };
                    return release;
                }),
            };
        }
        catch (e) {
            this.handleGenericErrors(e);
        }
    }
}
exports.GitlabReleasesDatasource = GitlabReleasesDatasource;
tslib_1.__decorate([
    (0, decorator_1.cache)({
        namespace: `datasource-${GitlabReleasesDatasource.id}`,
        key: ({ registryUrl, packageName }) => 
        // TODO: types (#22198)
        `${registryUrl}/${packageName}`,
    })
], GitlabReleasesDatasource.prototype, "getReleases", null);
//# sourceMappingURL=index.js.map